/*
 * linux/sound/soc/codecs/tlv320aic32x4.c
 *
 *
 * Copyright (C) 2011 Texas Instruments, Inc.
 *
 * Based on sound/soc/codecs/wm8753.c by Liam Girdwood
 *
 * This package is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * THIS PACKAGE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 * History:
 *
 * Rev 0.1   ASoC driver support    Mistral      31-04-2009
 * The AIC32 ASoC driver is ported for the codec AIC32x4.
 *
 *
 * Rev 1.0   Mini DSP support    Mistral         11-05-2009
 * Added mini DSP programming support
 *
 * Rev 1.1   Mixer controls   Mistral            18-01-2011
 * Added all the possible mixer controls.
 *
 *
 *
 * Rev 1.2   Additional Codec driver support     Mistral     2-02-2011
 * Support for AIC3253,AIC3206,AIC3256         
 */

/*
 * Includes
 */
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/pm.h>
#include <linux/i2c.h>
#include <linux/gpio.h>
#include <linux/debugfs.h>
#include <linux/regulator/consumer.h>
#include <linux/platform_device.h>
#include <linux/slab.h>
#include <linux/cdev.h>

#include <sound/core.h>
#include <sound/pcm.h>
#include <sound/pcm_params.h>
#include <sound/soc.h>
#include <sound/soc-dapm.h>
#include <sound/initval.h>

#include "tlv320aic32x4.h"
#include "crestron/crest_audio.h"
#include <linux/crestron.h>


/*
 * enable debug prints in the driver
 */
//#define CONFIG_SND_SOC_TLV320AIC32x4_DEBUG 1   //RJK enabled
#ifdef CONFIG_SND_SOC_TLV320AIC32x4_DEBUG
#define dprintk(x...)   printk(x)
#else
#define dprintk(x...)
#endif

//RJK
//#define RJK_DEBUG_FS 1
#undef RJK_DEBUG_FS

#ifdef RJK_DEBUG_FS
struct dentry *dbgfile;
struct dentry *dbgfile_u64;
struct dentry *dbgroot;
char ker_buf[10];
u64 rjk_debug_u64;
u32 g_dbg_rate=0, g_dbg_mclk=0;
struct rjk_debug_data {
	u32 data1;
	u32 data2;
	u32 data3;
	u32 data4;
	u32 data5;
};

static struct rjk_debug_data rjk_shared_data;
#endif


/*
 * Macros
 */
extern int aic32x4_minidsp_program(struct snd_soc_codec *codec);
extern void aic32x4_add_minidsp_controls(struct snd_soc_codec *codec);

#ifdef AIC3xxx_TiLoad
extern int aic3xxx_driver_init(struct snd_soc_codec *codec);
#endif

/* crestron UIHal interface */
extern int uiAudioInit(struct snd_soc_codec *codec);

/* User defined Macros kcontrol builders */
#define SOC_SINGLE_AIC32x4(xname)                                       \
        {                                                               \
                .iface = SNDRV_CTL_ELEM_IFACE_MIXER, .name = xname,     \
                        .info = __new_control_info, .get = __new_control_get, \
                        .put = __new_control_put,                       \
                        .access = SNDRV_CTL_ELEM_ACCESS_READWRITE,      \
                        }

#define SOC_DOUBLE_R_AIC32x4(xname, min, max, shift, mask, invert) \
        {	.iface = SNDRV_CTL_ELEM_IFACE_MIXER, .name = (xname),   \
                        .info = snd_soc_info_volsw_2r_aic32x4,          \
                        .get = snd_soc_get_volsw_2r_aic32x4, .put = snd_soc_put_volsw_2r_aic32x4, \
                        .private_value = (signed char)(max) | ((shift) << 8)  | \
                        ((mask) << 12) | ((invert) << 20) | ((signed char)(min) << 24) }

/*
*****************************************************************************
* Function Prototype
*****************************************************************************
*/
u8 aic32x4_read(struct snd_soc_codec *codec, u16 reg);
int aic32x4_write(struct snd_soc_codec *codec, u16 reg, u8 value);
inline void aic32x4_write_reg_cache(struct snd_soc_codec *codec,
                                    u16 reg, u8 value);
int aic32x4_reset_cache (struct snd_soc_codec *codec);
static int aic32x4_hw_params(struct snd_pcm_substream *substream,
			     struct snd_pcm_hw_params *params,
                             struct snd_soc_dai *);
int aic32x4_mute(struct snd_soc_dai *dai, int mute);
static int aic32x4_set_dai_sysclk(struct snd_soc_dai *codec_dai,
				  int clk_id, unsigned int freq, int dir);
static int aic32x4_set_dai_fmt(struct snd_soc_dai *codec_dai, unsigned int fmt);
static int aic32x4_set_bias_level(struct snd_soc_codec *codec, enum snd_soc_bias_level level);
static int __new_control_info(struct snd_kcontrol *kcontrol,
			      struct snd_ctl_elem_info *uinfo);
static int __new_control_get(struct snd_kcontrol *kcontrol,
			     struct snd_ctl_elem_value *ucontrol);
static int __new_control_put(struct snd_kcontrol *kcontrol,
			     struct snd_ctl_elem_value *ucontrol);
static int snd_soc_info_volsw_2r_aic32x4(struct snd_kcontrol *kcontrol,
                                         struct snd_ctl_elem_info *uinfo);
static int snd_soc_get_volsw_2r_aic32x4(struct snd_kcontrol *kcontrol,
					struct snd_ctl_elem_value *ucontrol);
static int snd_soc_put_volsw_2r_aic32x4(struct snd_kcontrol *kcontrol,
					struct snd_ctl_elem_value *ucontrol);

/*
*****************************************************************************
* Global Variable
*****************************************************************************
*/
static u8 aic32x4_reg_ctl;

/* whenever aplay/arecord is run, aic32x4_hw_params() function gets called.
 * This function reprograms the clock dividers etc. this flag can be used to
 * disable this when the clock dividers are programmed by pps config file
 */
static int soc_static_freq_config = 1;  //default
//RJK static int soc_static_freq_config = 0;

/*
*****************************************************************************
* Structure Declaration
*****************************************************************************
*/
static struct snd_soc_device *aic32x4_socdev;


static const char *mute[] = { "Unmute", "Mute" };
static const char *dacvolume_extra[] =
{ "L&R Ind Vol", "LVol=RVol", "RVol=LVol" };
static const char *dacsoftstep_control[] =
{ "1 step/sample", "1 step/2 sample", "disabled" };
static const char *headset_detection[] = { "Disabled", "Enabled" };
static const char *drc_enable[] = { "Disabled", "Enabled" };
static const char *beep_generator[] = { "Disabled", "Enabled" };
static const char *beepvolume_extra[] =
{ "L&R Ind Vol", "LVol=RVol", "RVol=LVol" };
static const char *micbias_voltage[] =
{ "1.04/1.25V", "1.425/1.7V", "2.075/2.5V", "POWER SUPPY" };
static const char *micpga_selection[] = { "off", "10k", "20k", "40k" };
static const char  *micpgagain_enable[] = { "Disabled", "Enabled" };
static const char *powerup[] = { "Power Down", "Power Up" };
static const char *vol_generation[] = { "AVDD", "LDOIN" };
static const char *path_control[] =
{ "Disabled", "LDAC Data", "RDAC Data", "L&RDAC Data" };
static const char  *agc_enable[] = { "Disabled", "Enabled" };

/*Creates an array of the Single Ended Widgets */
static const struct soc_enum aic32x4_enum[] = {
        SOC_ENUM_SINGLE (DAC_MUTE_CTRL_REG, 3, 2, mute),
        SOC_ENUM_SINGLE (DAC_CHN_REG, 0, 3, dacsoftstep_control),
        SOC_ENUM_SINGLE (DAC_MUTE_CTRL_REG, 0, 4, dacvolume_extra),
        SOC_ENUM_SINGLE (HEADSET_DETECT, 7, 2, headset_detection),
        SOC_ENUM_SINGLE (ADC_FGA, 7, 2, mute),
        SOC_ENUM_DOUBLE (DRC_CTRL_REG1, 6, 5, 2, drc_enable),
        SOC_ENUM_SINGLE (BEEP_CTRL_REG1, 7, 2, beep_generator),
        SOC_ENUM_SINGLE (BEEP_CTRL_REG2, 6, 3, beepvolume_extra),
        SOC_ENUM_SINGLE (MICBIAS_CTRL, 4, 4, micbias_voltage),

#ifndef AIC3253_CODEC_SUPPORT
        SOC_ENUM_SINGLE (LMICPGA_PIN_CFG, 6, 4, micpga_selection),
        SOC_ENUM_SINGLE (LMICPGA_PIN_CFG, 4, 4, micpga_selection),
        SOC_ENUM_SINGLE (LMICPGA_PIN_CFG, 2, 4, micpga_selection),
        SOC_ENUM_SINGLE (LMICPGA_PIN_CFG, 0, 4, micpga_selection),
        SOC_ENUM_SINGLE (LMICPGA_NIN_CFG, 6, 4, micpga_selection),
        SOC_ENUM_SINGLE (LMICPGA_NIN_CFG, 4, 4, micpga_selection),
        SOC_ENUM_SINGLE (LMICPGA_NIN_CFG, 2, 4, micpga_selection),
        SOC_ENUM_SINGLE (LMICPGA_NIN_CFG, 0, 4, micpga_selection),
        SOC_ENUM_SINGLE (RMICPGA_PIN_CFG, 6, 4, micpga_selection),
        SOC_ENUM_SINGLE (RMICPGA_PIN_CFG, 4, 4, micpga_selection),
        SOC_ENUM_SINGLE (RMICPGA_PIN_CFG, 2, 4, micpga_selection),
        SOC_ENUM_SINGLE (RMICPGA_PIN_CFG, 0, 4, micpga_selection),
        SOC_ENUM_SINGLE (RMICPGA_NIN_CFG, 6, 4, micpga_selection),
        SOC_ENUM_SINGLE (RMICPGA_NIN_CFG, 4, 4, micpga_selection),
        SOC_ENUM_SINGLE (RMICPGA_NIN_CFG, 2, 4, micpga_selection),
        SOC_ENUM_SINGLE (RMICPGA_NIN_CFG, 0, 4, micpga_selection),
        SOC_ENUM_SINGLE (LMICPGA_VOL_CTRL, 7, 2, micpgagain_enable),
        SOC_ENUM_SINGLE (RMICPGA_VOL_CTRL, 7, 2, micpgagain_enable),
#endif

	SOC_ENUM_SINGLE (MICBIAS_CTRL, 6, 2, powerup),    //RJK FYI, micbias INT is not connected on tst600
	SOC_ENUM_SINGLE (MICBIAS_CTRL, 3, 2, vol_generation),

#ifndef AIC3253_CODEC_SUPPORT
	SOC_ENUM_SINGLE (ADC_CHN_REG, 7, 2, powerup),
	SOC_ENUM_SINGLE (ADC_CHN_REG, 6, 2, powerup),
#endif

	SOC_ENUM_SINGLE (DAC_CHN_REG, 7, 2, powerup),
	SOC_ENUM_SINGLE (DAC_CHN_REG, 6, 2, powerup),
        SOC_ENUM_SINGLE (DAC_MUTE_CTRL_REG, 2, 2, mute),

#ifndef AIC3253_CODEC_SUPPORT
        SOC_ENUM_SINGLE (ADC_FGA, 3, 2, mute),
#endif

        SOC_ENUM_SINGLE (DAC_CHN_REG, 4, 4, path_control),
        SOC_ENUM_SINGLE (DAC_CHN_REG, 2, 4, path_control),
	SOC_ENUM_DOUBLE (OUT_PWR_CTRL, 5, 4, 2, powerup),
	SOC_ENUM_DOUBLE (OUT_PWR_CTRL, 3, 2, 2, powerup),
	SOC_ENUM_DOUBLE (OUT_PWR_CTRL, 1, 0, 2, powerup),
	SOC_ENUM_SINGLE (HPL_GAIN, 6, 2, mute),
	SOC_ENUM_SINGLE (HPR_GAIN, 6, 2, mute),

#ifndef AIC3253_CODEC_SUPPORT
	SOC_ENUM_SINGLE (LOL_GAIN, 6, 2, mute),
	SOC_ENUM_SINGLE (LOR_GAIN, 6, 2, mute),
	SOC_ENUM_SINGLE (LEFT_AGC_REG1, 7, 2, agc_enable),
	SOC_ENUM_SINGLE (RIGHT_AGC_REG1, 7, 2, agc_enable),
#endif
};

/*
 * Structure Initialization
 */
/* Various Controls For AIC32x4 */
static const struct snd_kcontrol_new aic32x4_snd_controls1[] = {

	/* Left DAC Mute Control */
        SOC_ENUM ("Left DAC Mute Control", aic32x4_enum[LDMUTE_ENUM]),

        /* Right DAC Mute Control */
        SOC_ENUM ("Right DAC Mute Control", aic32x4_enum[RDMUTE_ENUM]),

	/* DAC Volume Soft Stepping Control */
        SOC_ENUM ("DAC Volume Soft Stepping", aic32x4_enum[SOFTSTEP_ENUM]),

        /* DAC Extra Volume Control */
        SOC_ENUM ("DAC Extra Volume Control", aic32x4_enum[DACEXTRA_ENUM]),

        /* Headset detection Enable/Disable control */
        SOC_ENUM ("Headset detection Enable/Disable", aic32x4_enum[HSET_ENUM]),

#ifndef AIC3253_CODEC_SUPPORT
        /* Left ADC Mute Control */
        SOC_ENUM ("Left ADC Mute Control", aic32x4_enum[LAMUTE_ENUM]),

        /* Right ADC Mute Control */
        SOC_ENUM ("Right ADC Mute Control", aic32x4_enum[RAMUTE_ENUM]),
#endif


	/* Mic Bias Voltage */
        SOC_ENUM ("Mic Bias Vol for CM=0.75/0.9 V", aic32x4_enum[MICBIAS_ENUM]),

#ifndef AIC3253_CODEC_SUPPORT
        /* IN1L to Left MICPGA Positive Terminal Selection */
        SOC_ENUM ("IN1L to LMICPGA +ve terminal sel",
                  aic32x4_enum[IN1L_LMICPGA_ENUM]),

        /* IN2L to Left MICPGA Positive Terminal Selection */
        SOC_ENUM ("IN2L to LMICPGA +ve terminal sel",
                  aic32x4_enum[IN2L_LMICPGA_ENUM]),

        /* IN3L to Left MICPGA Positive Terminal Selection */
        SOC_ENUM ("IN3L to LMICPGA +ve terminal sel",
                  aic32x4_enum[IN3L_LMICPGA_ENUM]),

        /* IN1R to Left MICPGA Positive Terminal Selection */
        SOC_ENUM ("IN1R to LMICPGA +ve terminal sel",
                  aic32x4_enum[IN1R_LMICPGA_ENUM]),

        /* CM1L to Left MICPGA Positive Terminal Selection */
        SOC_ENUM ("CM1L to LMICPGA +ve terminal sel",
                  aic32x4_enum[CM1L_LMICPGA_ENUM]),

        /* IN2R to Left MICPGA Positive Terminal Selection */
        SOC_ENUM ("IN2R to LMICPGA +ve terminal sel",
                  aic32x4_enum[IN2R_LMICPGA_ENUM]),

        /* IN3R to Left MICPGA Positive Terminal Selection */
        SOC_ENUM ("IN3R to LMICPGA +ve terminal sel",
                  aic32x4_enum[IN3R_LMICPGA_ENUM]),

        /*CM2L to Left MICPGA Positive Terminal Selection */
        SOC_ENUM ("IN1L to LMICPGA +ve terminal sel",
                  aic32x4_enum[CM2L_LMICPGA_ENUM]),
#endif
};

/* Various Controls For AIC32x4 */
static const struct snd_kcontrol_new aic32x4_snd_controls2[] = {

	/* IN1L to HPL Volume Control */
        SOC_SINGLE ("IN1L to HPL volume control", IN1L_HPL_CTRL, 0, 0x72, 0),

        /* IN1R to HPR Volume Control */
        SOC_SINGLE ("IN1R to HPR volume control", IN1R_HPR_CTRL, 0, 0x72, 0),

	/* Mic Bias Power up/down */
        SOC_ENUM ("Mic Bias Power up/down", aic32x4_enum[MICBIASPOWER_ENUM]),

	/* Mic Bias Generation */
        SOC_ENUM ("Mic Bias Voltage Generation",aic32x4_enum[MICBIASVOLTAGE_ENUM]),

	/* IN1L to HPL routing */
        SOC_SINGLE ("IN1L to HPL Route", HPL_ROUTE_CTRL, 2, 1, 0),

#ifndef AIC3253_CODEC_SUPPORT
        /* MAL output to HPL */
        SOC_SINGLE ("MAL Output to HPL Route", HPL_ROUTE_CTRL, 1, 1, 0),

        /*MAR output to HPL */
        SOC_SINGLE ("MAR Output to HPL Route", HPL_ROUTE_CTRL, 0, 1, 0),
#endif

        /* IN1R to HPR routing */
        SOC_SINGLE ("IN1R to HPR Route", HPR_ROUTE_CTRL, 2, 1, 0),

#ifndef AIC3253_CODEC_SUPPORT
        /* MAR to HPR routing */
        SOC_SINGLE ("MAR Output to HPR Route", HPR_ROUTE_CTRL, 1, 1, 0),
#endif

        /* HPL Output to HRP routing */
        SOC_SINGLE ("HPL Output to HPR Route", HPR_ROUTE_CTRL, 0, 1, 0),

#ifndef AIC3253_CODEC_SUPPORT
        /* MAL Output to LOL routing*/
        SOC_SINGLE ("MAL Output to LOL Route", LOL_ROUTE_CTRL, 1, 1, 0),

        /* LOR Output to LOL routing*/
        SOC_SINGLE ("LOR Output to LOL Route", LOL_ROUTE_CTRL, 0, 1, 0),

        /* MAR Output to LOR routing*/
        SOC_SINGLE ("MAR Outout to LOR Route", LOR_ROUTE_CTRL, 1, 1, 0),
#endif

	/* DRC Threshold value  Control */
        SOC_SINGLE ("DRC Threshold value (0=-3db, 7=-24db)", DRC_CTRL_REG1,
                    2, 0x07, 0),

        /* DRC Hysteresis value control */
        SOC_SINGLE ("DRC Hysteresis value (0=0db, 3=3db)", DRC_CTRL_REG1,
                    0, 0x03, 0),

        /* DRC Hold time control */
        SOC_SINGLE ("DRC hold time", DRC_CTRL_REG2, 3, 0x0F, 0),

        /* DRC Attack rate control */
        SOC_SINGLE ("DRC attack rate", DRC_CTRL_REG3, 4, 0x0F, 0),

        /* DRC Decay rate control */
        SOC_SINGLE ("DRC decay rate", DRC_CTRL_REG3, 0, 0x0F, 0),

        /* Beep Length MSB control */
        SOC_SINGLE ("Beep Length MSB", BEEP_CTRL_REG3, 0, 255, 0),

        /* Beep Length MID control */
        SOC_SINGLE ("Beep Length MID", BEEP_CTRL_REG4, 0, 255, 0),

        /* Beep Length LSB control */
        SOC_SINGLE ("Beep Length LSB", BEEP_CTRL_REG5, 0, 255, 0),

        /* Beep Sin(x) MSB control */
        SOC_SINGLE ("Beep Sin(x) MSB", BEEP_CTRL_REG6, 0, 255, 0),

        /* Beep Sin(x) LSB control */
        SOC_SINGLE ("Beep Sin(x) LSB", BEEP_CTRL_REG7, 0, 255, 0),

        /* Beep Cos(x) MSB control */
        SOC_SINGLE ("Beep Cos(x) MSB", BEEP_CTRL_REG8, 0, 255, 0),

        /* Beep Cos(x) LSB control */
        SOC_SINGLE ("Beep Cos(x) LSB", BEEP_CTRL_REG9, 0, 255, 0),
};

/* Various Controls For AIC32x4 */
static const struct snd_kcontrol_new aic32x4_snd_controls3[] = {

        /* Left DAC Power Control */
        SOC_ENUM ("Left DAC channel Power up/down", aic32x4_enum[LDACPOWER_ENUM]),

	/* Right DAC Power Control */
        SOC_ENUM ("Right DAC channel Power up/down", aic32x4_enum[RDACPOWER_ENUM]),

        /* Left DAC Data Path Control */
        SOC_ENUM ("Left DAC Data Path Control", aic32x4_enum[LDACPATH_ENUM]),

	/* Right DAC Data Path Control */
        SOC_ENUM ("Right DAC Data Path Control", aic32x4_enum[RDACPATH_ENUM]),

        /* Left/Right DAC Digital Volume Control */
        SOC_DOUBLE_R_AIC32x4 ("Left/Right DAC Digital Volume Control",
                              0, 100, 0, 0x64, 0),


#ifndef AIC3253_CODEC_SUPPORT
	/* Left ADC Power Control */
        SOC_ENUM ("Left ADC channel Power up/down", aic32x4_enum[LADCPOWER_ENUM]),

	/* Right ADC Power Control */
        SOC_ENUM ("Right ADC channel Power up/down", aic32x4_enum[RADCPOWER_ENUM]),

        /* Left/Right ADC Fine Gain Adjust */
	SOC_DOUBLE ("Left/Right ADC Fine Gain Adjust", ADC_FGA, 4, 0, 0x04, 0),

        /* Left/Right ADC Volume Control */
        SOC_DOUBLE_R_AIC32x4 ("Left/Right ADC Volume Control",
                              -100, 100, 0, 0x3F, 0),
#endif

        /* HeadPhone Driver Power Control */
        SOC_ENUM ("HP Power up/down", aic32x4_enum[HPPOWER_ENUM]),

        /*Line-Out Driver Power Control */
        SOC_ENUM ("LO Power up/down", aic32x4_enum[LOPOWER_ENUM]),

        /* Mixer Amplifiers Power Control */
        SOC_ENUM ("Mixer Amplifier Power up/down", aic32x4_enum[MAPOWER_ENUM]),

        /* Left HP Driver Mute Control */
        SOC_ENUM ("Left HP driver mute", aic32x4_enum[LHPMUTE_ENUM]),

        /* Right HP Driver Mute Control */
        SOC_ENUM ("Right HP driver mute", aic32x4_enum[RHPMUTE_ENUM]),

#ifndef AIC3253_CODEC_SUPPORT
        /* Left LO Driver Mute control */
        SOC_ENUM ("Left LO driver mute", aic32x4_enum[LLOMUTE_ENUM]),

	/* Right LO Driver Mute control */
        SOC_ENUM ("Right LO driver mute", aic32x4_enum[RLOMUTE_ENUM]),
#endif
#if 0 //for headphone support
	/*HP Driver Gain Control*/
        SOC_DOUBLE_R_AIC32x4("HP Driver Gain", HPL_GAIN, HPR_GAIN, 0, 0x23, 0),
#endif
#ifndef AIC3253_CODEC_SUPPORT
#if 0 //unused on crestron panels 
       /*LO Driver Gain Control*/
        SOC_DOUBLE_R_AIC32x4("LO Driver Gain", LOL_GAIN, LOR_GAIN, 0, 0x23 , 0),
#endif


	/* Mixer Amplifier Volume Control */
        SOC_DOUBLE_R ("Mixer Amplier Volume Control",
                      MAL_CTRL_REG, MAR_CTRL_REG, 0, 0x50, 0),


        /*Left/Right MICPGA Volume Control */
        SOC_DOUBLE_R ("Let/Right MICPGA Volume Control",
                      LMICPGA_VOL_CTRL, RMICPGA_VOL_CTRL, 0, 0x5F, 0),

#endif

        /* Beep generator Volume Control */
        SOC_DOUBLE_R ("Beep generator Volume Control (0=0dB, 63=-63dB)",
                      BEEP_CTRL_REG1, BEEP_CTRL_REG2, 0, 0x3F, 1),

        /* Left Audio gain control (AGC) Enable/Disable Control */
        SOC_ENUM ("Left AGC Enable/Disable", aic32x4_enum[LAGC_ENUM]),

#ifndef AIC3253_CODEC_SUPPORT
        /* Left/Right Audio gain control (AGC) Enable/Disable Control */
        SOC_ENUM ("Right AGC Enable/Disable", aic32x4_enum[RAGC_ENUM]),

        /* Left/Right AGC Target level control */
        SOC_DOUBLE_R (" Left/Right AGC Target Level Control",
                      LEFT_AGC_REG1, RIGHT_AGC_REG1, 4, 0x07, 1),

        /* Left/Right AGC Hysteresis Control */
        SOC_DOUBLE_R ("Left/Right AGC Hysteresis Control",
                      LEFT_AGC_REG1, RIGHT_AGC_REG1, 0, 0x03, 0),

        /*Left/Right AGC Maximum PGA applicable */
        SOC_DOUBLE_R ("Left/Right AGC Maximum PGA Control",
                      LEFT_AGC_REG3,RIGHT_AGC_REG3,  0, 0x7F, 0),

        /* Left/Right AGC Noise Threshold */
        SOC_DOUBLE_R ("Left/Right AGC Noise Threshold",
                      LEFT_AGC_REG2, RIGHT_AGC_REG2, 1, 0x1F, 1),

        /* Left/Right AGC Attack Time control */
        SOC_DOUBLE_R ("Left/Right AGC Attack Time control",
                      LEFT_AGC_REG4, RIGHT_AGC_REG4, 3, 0x1F, 0),

        /* Left/Right AGC Decay Time control */
        SOC_DOUBLE_R ("LEft/Right AGC Decay Time control",
                      LEFT_AGC_REG5, RIGHT_AGC_REG5, 3, 0x1F, 0),

        /* Left/Right AGC Noise Debounce control */
        SOC_DOUBLE_R ("LEft/Right AGC Noice bounce control",
                      LEFT_AGC_REG6, RIGHT_AGC_REG6, 0, 0x1F, 0),

        /* Left/Right AGC Signal Debounce control */
        SOC_DOUBLE_R ("Left/Right AGC Signal bounce control",
                      LEFT_AGC_REG7, RIGHT_AGC_REG7, 0, 0x0F, 0),
#endif

	/* DAC Signal Processing Block Control*/
	SOC_SINGLE ("DAC PRB Selection(1 to 25)", DAC_PRB,  0, 0x19, 0),
	
	/* ADC Signal Processing Block Control */
	SOC_SINGLE ("ADC PRB Selection(1 to 18)", ADC_PRB,  0, 0x12, 0),

        /* sound new kcontrol for Programming the registers from user space */
        SOC_SINGLE_AIC32x4("Program Registers"),
};

/* Various Controls For AIC32x4 */
static const struct snd_kcontrol_new aic32x4_snd_controls4[] = {

#ifndef AIC3253_CODEC_SUPPORT
        /* IN1R to Right MICPGA Positive Terminal Selection */
        SOC_ENUM ("IN1R to RMICPGA +ve terminal sel",
                  aic32x4_enum[IN1R_RMICPGA_ENUM]),

        /* IN2R to Right MICPGA Positive Terminal Selection */
        SOC_ENUM ("IN2R to RMICPGA +ve terminal sel",
                  aic32x4_enum[IN2R_RMICPGA_ENUM]),

        /* IN3R to Right MICPGA Positive Terminal Selection */
        SOC_ENUM ("IN3R to RMICPGA +ve terminal sel",
                  aic32x4_enum[IN3R_RMICPGA_ENUM]),

        /* IN2L to Right MICPGA Positive Terminal Selection */
        SOC_ENUM ("IN2L to RMICPGA +ve terminal sel",
                  aic32x4_enum[IN2L_RMICPGA_ENUM]),

        /* CM1R to Right MICPGA Positive Terminal Selection */
        SOC_ENUM ("CM1R to RMICPGA +ve terminal sel",
                  aic32x4_enum[CM1R_RMICPGA_ENUM]),

        /* IN1L to Right MICPGA Positive Terminal Selection */
        SOC_ENUM ("IN1L to RMICPGA +ve terminal sel",
                  aic32x4_enum[IN1L_RMICPGA_ENUM]),

        /* IN3L to Right MICPGA Positive Terminal Selection */
        SOC_ENUM ("IN3L to RMICPGA +ve terminal sel",
                  aic32x4_enum[IN3L_RMICPGA_ENUM]),

        /* CM2R to Right MICPGA Positive Terminal Selection */
        SOC_ENUM ("CM2R to RMICPGA +ve terminal sel",
                  aic32x4_enum[CM2R_RMICPGA_ENUM]),


        /* Left MICPGA Gain Enabled/Disable */
        SOC_ENUM ("Left MICPGA Gain Enable", aic32x4_enum[LMICPGAGAIN_ENUM]),

        /* Right MICPGA Gain Enabled/Disable */
        SOC_ENUM ("Right MICPGA Gain Enable", aic32x4_enum[RMICPGAGAIN_ENUM]),
#endif

        /* DRC Enable/Disable Control */
        SOC_ENUM ("DRC Enable/Disable", aic32x4_enum[DRC_ENUM]),

        /* Beep generator Enable/Disable control */
        SOC_ENUM ("Beep generator Enable/Disable", aic32x4_enum[BEEP_ENUM]),

        /* Beep Master Volume Control */
        SOC_ENUM ("Beep Master Volume Control", aic32x4_enum[BEEPEXTRA_ENUM]),


#if defined(AIC3206_CODEC_SUPPORT) || defined( AIC3256_CODEC_SUPPORT )

        /*charge pump configuration for n/8 peak load current*/
        SOC_SINGLE("Charge pump configuration for n/8 peak load current", CHRG_CTRL_REG, 4, 8, 0),
       
        /*charge pump clock divide control*/
        SOC_SINGLE("charge pump clock divide control", CHRG_CTRL_REG, 0, 16, 0),  
       

        /*HPL, HPR master gain control in ground centerd mode */
        SOC_SINGLE("HPL & HPR master gain control in ground centered mode", HED_CONF_REG, 5, 4, 0),

        /*headphone amplifier compensation adjustment */
        SOC_SINGLE(" headphone amplifier compensation adjustment", HED_CONF_REG, 7, 1, 0),

        /*Disable/Enable ground centered mode*/
        SOC_SINGLE(" Disable/Enable ground centered mode", HED_CONF_REG, 4, 1, 0),

        /*headphone driver power configuration*/
        SOC_SINGLE(" Headphone driver power configration", HED_CONF_REG, 2, 4, 0),

        /*DC offset correction*/
        SOC_SINGLE(" DC offset correction", HED_CONF_REG, 0, 4, 0),

#endif 


};

/* The sturcture contains the different values for mclk */
static const struct aic32x4_rate_divs aic32x4_divs[] = {
/*
 * mclk, rate, p_val, pll_j, pll_d, dosr, ndac, mdac, aosr, nadc, madc, blck_N,
 * codec_speficic_initializations
 */
	/* 8k rate */
	{24000000, 8000, 1, 4, 960, 768, 2, 8, 128, 8, 12, 2,
	 {{60, 0}, {61, 0}}},
};

struct uiAudio *uiAudio_data;

/* read function used for Crestron audio UI interface */
/* wrapped the aic32x4_read function so the function could remain static */
int ui_aic32x4_read(struct snd_soc_codec *codec, u16 reg)
{
	int ret;

	ret=aic32x4_read(codec, reg);
	return ret;
}

/* write function used for Crestron audio UI interface */
/* UI interface will send value to this function and it will be 
 * checked, and the correct MACRO will be called for functionality
 */
int ui_aic32x4_write(struct snd_soc_codec *codec, u8 param, u32 value)
{
        int ret;
#if 0
/* LOL unmute and gain 0db */
        {LOL_GAIN, 0x0},
        /* LOR unmute and gain 0db */
        {LOR_GAIN, 0x0},
       /* Unmute DAC Left and Right channels */
        {DAC_MUTE_CTRL_REG, 0x00},
        /* LDAC VOLUME */
        {LDAC_VOL, 0x0A},
        {RDAC_VOL, 0x0A},
#endif

	switch(param)
	{
		case PLAY_CAPABLE:
			printk(KERN_ERR "setting PLAY_CAPABLE to %d\n", value);
			break;
		case REC_CAPABLE:
			printk(KERN_ERR "setting REC_CAPABLE to %d\n", value);
			break;
		case PLAY_VOL_DEFAULT:
			printk(KERN_ERR "setting PLAY_VOL_CAPABLE to %d\n", value);
			break;
		case REC_BOOST_CAPABLE:
			printk(KERN_ERR "setting PLAY_VOL_CAPABLE to %d\n", value);
			break;
		case REC_BOOST_MIN:
			printk(KERN_ERR "setting PLAY_VOL_CAPABLE to %d\n", value);
			break;
		case REC_BOOST_MAX:
			printk(KERN_ERR "setting PLAY_VOL_CAPABLE to %d\n", value);
			break;
		case AEC_CAPABLE:
			printk(KERN_ERR "setting PLAY_VOL_CAPABLE to %d\n", value);
			break;
		case AEC_DEFAULT:
			printk(KERN_ERR "setting PLAY_VOL_CAPABLE to %d\n", value);
			break;
		case PLAY_AGC_CAPABLE:
			printk(KERN_ERR "setting PLAY_VOL_CAPABLE to %d\n", value);
			break;
		case PLAY_AGC_DEFAULT:
			printk(KERN_ERR "setting PLAY_VOL_CAPABLE to %d\n", value);
			break;
		case REC_AGC_CAPABLE:
			printk(KERN_ERR "setting PLAY_VOL_CAPABLE to %d\n", value);
			break;
		case REC_AGC_DEFAULT:
			printk(KERN_ERR "setting PLAY_VOL_CAPABLE to %d\n", value);
			break;
		case PLAY_VOL:
			printk(KERN_ERR "setting PLAY_VOL_CAPABLE to %d\n", value);
			break;
		case REC_BOOST_LEVEL:
			printk(KERN_ERR "setting PLAY_VOL_CAPABLE to %d\n", value);
			break;
		case PLAY_MUTE:
			printk(KERN_ERR "setting PLAY_MUTE: to %d\n", value);
			//ret=aic32x4_write(codec, DAC_MUTE_CTRL_REG, value);
			ret=aic32x4_mute(codec->dai, value);
			break;
		case REC_MUTE:
			printk(KERN_ERR "setting PLAY_VOL_CAPABLE to %d\n", value);
			break;
		default:
			printk(KERN_ERR "invalid REG address %d val=%d\n", param, value);
			break;
	}
        return ret;
}


#ifdef RJK_DEBUG_FS
static int dbg_freq_open_file(struct inode *inode, struct file *file)
{
        file->private_data = inode->i_private;
        return 0;
}

static ssize_t dbg_freq_read_file(struct file *file, char __user *user_buf,
                               size_t count, loff_t *position)
{
	ssize_t ret;
        struct snd_soc_codec *codec = file->private_data;
	struct aic32x4_priv *aic32x4 = codec->private_data;
        char *buf = kmalloc(PAGE_SIZE, GFP_KERNEL);
        if (!buf)
                return -ENOMEM;

	printk(KERN_ERR " freq is set to: 0x%x\(%d\)\n", *user_buf, aic32x4->sysclk, aic32x4->sysclk);
	ret = simple_read_from_buffer(user_buf, count, position, buf, ret);
        kfree(buf);
	return ret;
}

static ssize_t dbg_freq_write_file(struct file *file,
                const char __user *user_buf, size_t count, loff_t *position)
{
 	char buf[32];
        int buf_size;
        char *start = buf;
        unsigned long reg, value;

	struct snd_soc_codec *codec = file->private_data;
        struct aic32x4_priv *aic32x4 = codec->private_data;

	printk(KERN_ERR "RJK: count=%d\n", count);
	buf_size = min(count, (sizeof(buf)-1));
	if (copy_from_user(buf, user_buf, buf_size))
                return -EFAULT;
        buf[buf_size] = 0;
	while (*start == ' ')
                start++;

	if (strict_strtoul(start, 16, &value))
                return -EINVAL;

	printk(KERN_ERR "value=0x%x \n", value);
	switch(value) {
		case 1: {
			aic32x4->sysclk = 0xb71b00;  // 12000000
			break;
		}
		case 2: {
			aic32x4->sysclk = 0x16e3600;  // 24000000
			break;
		}
		default:
			printk(KERN_ERR "not a valid value, only 1 or 2 will be accepted\n");
			printk(KERN_ERR "1=120000000 : 2=24000000\n");
			printk(KERN_ERR "Try Again !\n");
	}
	printk(KERN_ERR "sysclk set to %d\n", aic32x4->sysclk);
}
static int dbg_rate_open_file(struct inode *inode, struct file *file)
{
        file->private_data = inode->i_private;
        return 0;
}

static ssize_t dbg_rate_read_file(struct file *file, char __user *user_buf,
                               size_t count, loff_t *position)
{
        ssize_t ret;
        struct snd_soc_codec *codec = file->private_data;
        struct aic32x4_priv *aic32x4 = codec->private_data;
        char *buf = kmalloc(PAGE_SIZE, GFP_KERNEL);
        if (!buf)
                return -ENOMEM;

        printk(KERN_ERR " debug rate is set to: 0x%x\(%d\)\n", g_dbg_rate, g_dbg_rate);
        ret = simple_read_from_buffer(user_buf, count, position, buf, ret);
        kfree(buf);
        return ret;
}

static ssize_t dbg_rate_write_file(struct file *file,
                const char __user *user_buf, size_t count, loff_t *position)
{
        char buf[32];
        int buf_size;
        char *start = buf;
        unsigned long reg, value;

        printk(KERN_ERR "RJK: count=%d\n", count);
        buf_size = min(count, (sizeof(buf)-1));
        if (copy_from_user(buf, user_buf, buf_size))
                return -EFAULT;
        buf[buf_size] = 0;
        while (*start == ' ')
                start++;

        if (strict_strtoul(start, 16, &value))
                return -EINVAL;

        printk(KERN_ERR "value=0x%x \n", value);
	g_dbg_rate=value;

        printk(KERN_ERR "rate set to %d\n", g_dbg_rate);
        printk(KERN_ERR "set to zero to go back to default behaviour\n", g_dbg_rate);
}
static const struct file_operations debug_freq_fops = {
        .open = dbg_freq_open_file,
        .read = dbg_freq_read_file,
        .write = dbg_freq_write_file,
};
static const struct file_operations debug_rate_fops = {
        .open = dbg_rate_open_file,
        .read = dbg_rate_read_file,
        .write = dbg_rate_write_file,
};

static void init_audio_debugfs(struct snd_soc_codec *codec)
{

	dbgroot = debugfs_create_dir("aic32x4", NULL); 
        if (!dbgroot) {
                printk(KERN_WARNING
                       "ASoC: Failed to create root debugfs directory\n");
                return;
        }
        dbgfile = debugfs_create_file("setfreq", 0644,
                                      dbgroot,
                                      codec, &debug_freq_fops);
        if (!dbgfile)
                printk(KERN_WARNING
                       "ASoC: Failed to setfreq debugfs file\n");

        dbgfile = debugfs_create_file("setrate", 0644,
                                      dbgroot,
                                      codec, &debug_rate_fops);
        if (!dbgfile)
                printk(KERN_WARNING
                       "ASoC: Failed to setfreq debugfs file\n");


}


static void cleanup_audio_debugfs(struct snd_soc_codec *codec)
{
        debugfs_remove_recursive(codec->debugfs_codec_root);
}

#else

static inline void init_audio_debugfs(struct snd_soc_codec *codec)
{
}

static inline void cleanup_audio_debugfs(struct snd_soc_codec *codec)
{
}
#endif


/*
 *----------------------------------------------------------------------------
 * @struct  snd_soc_codec_dai_ops |
 *          It is SoC Codec DAI Operations structure
 *----------------------------------------------------------------------------
 */
static struct snd_soc_dai_ops tlv32x4_dai_ops = {
        .hw_params = aic32x4_hw_params,
        .digital_mute = aic32x4_mute,
        .set_sysclk = aic32x4_set_dai_sysclk,
        .set_fmt = aic32x4_set_dai_fmt,
};

/*
 *----------------------------------------------------------------------------
 * @struct  snd_soc_codec_dai |
 *          It is SoC Codec DAI structure which has DAI capabilities viz.,
 *          playback and capture, DAI runtime information viz. state of DAI
 *			and pop wait state, and DAI private data.
 *          The AIC32x4 rates ranges from 8k to 192k
 *          The PCM bit format supported are 16, 20, 24 and 32 bits
 *----------------------------------------------------------------------------
 */
struct snd_soc_dai tlv320aic32x4_dai = {
	.name = "aic32x4",
	.playback = {
                .stream_name = "Playback",
                .channels_min = 1,
                .channels_max = 2,
                .rates = AIC32x4_RATES,
                .formats = AIC32x4_FORMATS,},
	.capture = {
                .stream_name = "Capture",
                .channels_min = 1,
                .channels_max = 2,
                .rates = AIC32x4_RATES,
                .formats = AIC32x4_FORMATS,},
        .ops = &tlv32x4_dai_ops,

};

EXPORT_SYMBOL_GPL(tlv320aic32x4_dai);

/*
*****************************************************************************
* Initializations
*****************************************************************************
*/

/*
 * AIC32x4 register cache
 * We are caching the registers here.
 * There is no point in caching the reset register.
 * NOTE: In AIC32x4, there are 127 registers supported in both page0 and page1
 *       The following table contains the page0 and page1 registers values.
 */
static const u8 aic32x4_reg[AIC32x4_CACHEREGNUM] = {
	0x00, 0x00, 0x50, 0x00,	/* 0 */
	0x00, 0x11, 0x04, 0x00,	/* 4 */
	0x00, 0x00, 0x00, 0x01,	/* 8 */
	0x01, 0x00, 0x80, 0x02,	/* 12 */
	0x00, 0x08, 0x01, 0x01,	/* 16 */
	0x80, 0x01, 0x00, 0x04,	/* 20 */
	0x00, 0x00, 0x01, 0x00,	/* 24 */
	0x00, 0x00, 0x01, 0x00,	/* 28 */
	0x00, 0x00, 0x00, 0x00,	/* 32 */
	0x00, 0x00, 0x00, 0x00,	/* 36 */
	0x00, 0x00, 0x00, 0x00,	/* 40 */
	0x00, 0x00, 0x00, 0x00,	/* 44 */
	0x00, 0x00, 0x00, 0x00,	/* 48 */
	0x00, 0x42, 0x02, 0x02,	/* 52 */
	0x42, 0x02, 0x02, 0x02,	/* 56 */
	0x00, 0x00, 0x00, 0x01,	/* 60 */
	0x01, 0x00, 0x14, 0x00,	/* 64 */
	0x0C, 0x00, 0x00, 0x00,	/* 68 */
	0x00, 0x00, 0x00, 0xEE,	/* 72 */
	0x10, 0xD8, 0x10, 0xD8,	/* 76 */
	0x00, 0x00, 0x88, 0x00,	/* 80 */
	0x00, 0x00, 0x00, 0x00,	/* 84 */
	0x7F, 0x00, 0x00, 0x00,	/* 88 */
	0x00, 0x00, 0x00, 0x00,	/* 92 */
	0x7F, 0x00, 0x00, 0x00,	/* 96 */
	0x00, 0x00, 0x00, 0x00,	/* 100 */
	0x00, 0x00, 0x00, 0x00,	/* 104 */
	0x00, 0x00, 0x00, 0x00,	/* 108 */
	0x00, 0x00, 0x00, 0x00,	/* 112 */
	0x00, 0x00, 0x00, 0x00,	/* 116 */
	0x00, 0x00, 0x00, 0x00,	/* 120 */
	0x00, 0x00, 0x00, 0x00,	/* 124 - PAGE0 Registers(127) ends here */
	0x01, 0x00, 0x08, 0x00,	/* 128, PAGE1-0 */
	0x00, 0x00, 0x00, 0x00,	/* 132, PAGE1-4 */
	0x00, 0x00, 0x00, 0x10,	/* 136, PAGE1-8 */
	0x00, 0x00, 0x00, 0x00,	/* 140, PAGE1-12 */
	0x40, 0x40, 0x40, 0x40,	/* 144, PAGE1-16 */
	0x00, 0x00, 0x00, 0x00,	/* 148, PAGE1-20 */
	0x00, 0x00, 0x00, 0x00,	/* 152, PAGE1-24 */
	0x00, 0x00, 0x00, 0x00,	/* 156, PAGE1-28 */
	0x00, 0x00, 0x00, 0x00,	/* 160, PAGE1-32 */
	0x00, 0x00, 0x00, 0x00,	/* 164, PAGE1-36 */
	0x00, 0x00, 0x00, 0x00,	/* 168, PAGE1-40 */
	0x00, 0x00, 0x00, 0x00,	/* 172, PAGE1-44 */
	0x00, 0x00, 0x00, 0x00,	/* 176, PAGE1-48 */
	0x00, 0x00, 0x00, 0x00,	/* 180, PAGE1-52 */
	0x00, 0x00, 0x00, 0x80,	/* 184, PAGE1-56 */
	0x80, 0x00, 0x00, 0x00,	/* 188, PAGE1-60 */
	0x00, 0x00, 0x00, 0x00,	/* 192, PAGE1-64 */
	0x00, 0x00, 0x00, 0x00,	/* 196, PAGE1-68 */
	0x00, 0x00, 0x00, 0x00,	/* 200, PAGE1-72 */
	0x00, 0x00, 0x00, 0x00,	/* 204, PAGE1-76 */
	0x00, 0x00, 0x00, 0x00,	/* 208, PAGE1-80 */
	0x00, 0x00, 0x00, 0x00,	/* 212, PAGE1-84 */
	0x00, 0x00, 0x00, 0x00,	/* 216, PAGE1-88 */
	0x00, 0x00, 0x00, 0x00,	/* 220, PAGE1-92 */
	0x00, 0x00, 0x00, 0x00,	/* 224, PAGE1-96 */
	0x00, 0x00, 0x00, 0x00,	/* 228, PAGE1-100 */
	0x00, 0x00, 0x00, 0x00,	/* 232, PAGE1-104 */
	0x00, 0x00, 0x00, 0x00,	/* 236, PAGE1-108 */
	0x00, 0x00, 0x00, 0x00,	/* 240, PAGE1-112 */
	0x00, 0x00, 0x00, 0x00,	/* 244, PAGE1-116 */
	0x00, 0x00, 0x00, 0x00,	/* 248, PAGE1-120 */
	0x00, 0x00, 0x00, 0x00	/* 252, PAGE1-124 */
};

/*
 * aic32x4 initialization data
 * This structure initialization contains the initialization required for
 * AIC32x4.
 * These registers values (reg_val) are written into the respective AIC32x4
 * register offset (reg_offset) to  initialize AIC32x4.
 * These values are used in aic32x4_init() function only.
 */
static const struct aic32x4_configs aic32x4_reg_init[] = {
	/* Carry out the software reset */
	{RESET, 0x01},
	/* Disable crude LDO */
	{POW_CFG, 0x00},


        #if defined(AIC3256_CODEC_SUPPORT) || defined( AIC3206_CODEC_SUPPORT)
        {PWR_CTRL_REG, 0x00},
        #endif
        
	#if !defined(AIC3206_CODEC_SUPPORT) || !defined(AIC3256_CODEC_SUPPORT)
  	/* Switch on the analog blocks */
	{LDO_CTL, 0x01},
        #endif

	/* Connect IN1_L and IN1_R to CM */
//RJK	{INPUT_CFG_REG, 0xc0},  //RJK- we do not have IN1,2,3R or IN1,2,3L connected//comment this

	/* PLL is CODEC_CLKIN */
	//{CLK_REG_1, MCLK_2_CODEC_CLKIN},  //RJK  seems it should be this but the PLLCLK below is correct.
	{CLK_REG_1, PLLCLK_2_CODEC_CLKIN},

	/* DAC_MOD_CLK is BCLK source */
	{AIS_REG_3, DAC_MOD_CLK_2_BDIV_CLKIN},
	/* Setting up DAC Channel */
	{DAC_CHN_REG,
	 LDAC_2_LCHN | RDAC_2_RCHN | SOFT_STEP_2WCLK},
	/* Headphone powerup */
//RJK	{HPHONE_STARTUP_CTRL, 0x35},
	/* Left Channel DAC recons filter's +ve terminal is routed to HPL */
//RJK	{HPL_ROUTE_CTRL, LDAC_CHNL_2_HPL},
	/* Right Channel DAC recons filter's +ve terminal is routed to HPR */
//RJK	{HPR_ROUTE_CTRL, RDAC_CHNL_2_HPR},

#ifndef AIC3253_CODEC_SUPPORT
        /* Left Channel DAC recons filter's +ve terminal is routed to LOL */
	{LOL_ROUTE_CTRL, LDAC_CHNL_2_HPL},
	/* Right Channel DAC recons filter's +ve terminal is routed to LOR */
	{LOR_ROUTE_CTRL, RDAC_CHNL_2_HPR},
#endif

	/* HPL unmute and gain 0db */
//RJK	{HPL_GAIN, 0x0},
	/* HPR unmute and gain 0db */
//RJK	{HPR_GAIN, 0x0},

#ifndef AIC3253_CODEC_SUPPORT
	/* LOL unmute and gain 0db */
	{LOL_GAIN, 0x0},
	/* LOR unmute and gain 0db */
	{LOR_GAIN, 0x0},
#endif
	/* Unmute DAC Left and Right channels */
	{DAC_MUTE_CTRL_REG, 0x00},  //RJK: DAC volume has independent channel control

	/* IN1_L is selected for left P */

#ifndef AIC3253_CODEC_SUPPORT
	{LMICPGA_PIN_CFG, 0x40},
	/* CM1 is selected for left M */
	{LMICPGA_NIN_CFG, 0x40},
	/* IN1_R is selected for right P */
	{RMICPGA_PIN_CFG, 0x40},
	/* CM1 is selected for right M */
	{RMICPGA_NIN_CFG, 0x40},
	/* Left mic PGA unmuted */
	{LMICPGA_VOL_CTRL, 0x00},
	/* Right mic PGA unmuted */
	{RMICPGA_VOL_CTRL, 0x00},
	/* ADC volume control change by 2 gain step per ADC Word Clock */
	{ADC_CHN_REG, 0x02},
	/* Unmute ADC left and right channels */
	{ADC_FGA, 0x00},
#endif
	/* LDAC VOLUME */
        {LDAC_VOL, 0x0A}, //each hex value equals .5 db   0x30=+24db=max
        {RDAC_VOL, 0x0A}, // 0x81=-63.5db=min 
	/* GPIO output is INT1 */  //RJK, may need for mic event.?
	{GPIO_CTRL, 0x14 },

	/* tst600 microphone */
	{MIS0_MFP4, 0x0e },
	{SCLK_MFP3, 0x02 },

	/*  Headset Insertion event will generate a INT1 interrupt */
	{INT1_CTRL, 0x80},  //RJK, no headphone but mic may go through same jack.
	/*Enable headset detection and button press with a debounce time 64ms */
	{HEADSET_DETECT, 0x89},
};

/* Left DAC_L Mixer */
static const struct snd_kcontrol_new hpl_output_mixer_controls[] = {
	SOC_DAPM_SINGLE("L_DAC switch", HPL_ROUTE_CTRL, 3, 1, 0),
	SOC_DAPM_SINGLE("IN1_L switch", HPL_ROUTE_CTRL, 2, 1, 0),
//	SOC_DAPM_SINGLE("Left_Bypass switch", HPL_ROUTE_CTL, 1, 1, 0),
};

/* Right DAC_R Mixer */
static const struct snd_kcontrol_new hpr_output_mixer_controls[] = {
	SOC_DAPM_SINGLE("R_DAC switch", HPR_ROUTE_CTRL, 3, 1, 0),
	SOC_DAPM_SINGLE("IN1_R switch", HPR_ROUTE_CTRL, 2, 1, 0),
//	SOC_DAPM_SINGLE("Right_Bypass switch", HPR_ROUTE_CTL, 1, 1, 0),
};

#ifndef AIC3253_CODEC_SUPPORT
static const struct snd_kcontrol_new lol_output_mixer_controls[] = {
	SOC_DAPM_SINGLE("L_DAC switch", LOL_ROUTE_CTRL, 3, 1, 0),
//	SOC_DAPM_SINGLE("Left_Bypass switch", HPL_ROUTE_CTL, 1, 1, 0),
};

static const struct snd_kcontrol_new lor_output_mixer_controls[] = {
	SOC_DAPM_SINGLE("R_DAC switch", LOR_ROUTE_CTRL, 3, 1, 0),
//	SOC_DAPM_SINGLE("Right_Bypass switch", LOR_ROUTE_CTL, 1, 1, 0),
};

/* Right DAC_R Mixer */
static const struct snd_kcontrol_new left_input_mixer_controls[] = {
	SOC_DAPM_SINGLE("IN1_L switch", LMICPGA_PIN_CFG, 6, 1, 0),
	SOC_DAPM_SINGLE("IN2_L switch", LMICPGA_PIN_CFG, 4, 1, 0),
	SOC_DAPM_SINGLE("IN3_L switch", LMICPGA_PIN_CFG, 2, 1, 0),
};

static const struct snd_kcontrol_new right_input_mixer_controls[] = {
	SOC_DAPM_SINGLE("IN1_R switch", RMICPGA_PIN_CFG, 6, 1, 0),
	SOC_DAPM_SINGLE("IN2_R switch", RMICPGA_PIN_CFG, 4, 1, 0),
	SOC_DAPM_SINGLE("IN3_R switch", RMICPGA_PIN_CFG, 2, 1, 0),
};
#endif

/* AIC32x4 Widget Structure */
static const struct snd_soc_dapm_widget aic32x4_dapm_widgets[] = {
	/* Left DAC to Left Outputs */
	/* dapm widget (stream domain) for left DAC */
	SND_SOC_DAPM_DAC("Left DAC", "Left Playback", DAC_CHN_REG, 7, 0),

	/* dapm widget (path domain) for left DAC_L Mixer */
	SND_SOC_DAPM_MIXER("HPL Output Mixer", SND_SOC_NOPM, 0, 0,
			   &hpl_output_mixer_controls[0],
			   ARRAY_SIZE(hpl_output_mixer_controls)),

 	/* dapm widget for Left Head phone Power */
	SND_SOC_DAPM_PGA("HPL Power", OUT_PWR_CTRL, 5, 0, NULL, 0),


#ifndef AIC3253_CODEC_SUPPORT
	/* dapm widget (path domain) for Left Line-out Output Mixer */
	SND_SOC_DAPM_MIXER("LOL Output Mixer", SND_SOC_NOPM, 0, 0,
			   &lol_output_mixer_controls[0],
			   ARRAY_SIZE(lol_output_mixer_controls)),

	/* dapm widget for Left Line-out Power */
	SND_SOC_DAPM_PGA("LOL Power", OUT_PWR_CTRL, 3, 0, NULL, 0),
#endif

	/* Right DAC to Right Outputs */
	/* dapm widget (stream domain) for right DAC */
	SND_SOC_DAPM_DAC("Right DAC", "Right Playback", DAC_CHN_REG, 6, 0),

	/* dapm widget (path domain) for right DAC_R mixer */
	SND_SOC_DAPM_MIXER("HPR Output Mixer", SND_SOC_NOPM, 0, 0,
			   &hpr_output_mixer_controls[0],
			   ARRAY_SIZE(hpr_output_mixer_controls)),
  
 	/* dapm widget for Right Head phone Power */
	SND_SOC_DAPM_PGA("HPR Power", OUT_PWR_CTRL, 4, 0, NULL, 0),

#ifndef AIC3253_CODEC_SUPPORT
	/* dapm widget for (path domain) Right Line-out Output Mixer */
	SND_SOC_DAPM_MIXER("LOR Output Mixer", SND_SOC_NOPM, 0, 0,
			   &lor_output_mixer_controls[0],
			   ARRAY_SIZE(lor_output_mixer_controls)),

 	/* dapm widget for Right Line-out Power */
	SND_SOC_DAPM_PGA("LOR Power", OUT_PWR_CTRL, 2, 0, NULL, 0),
#endif

	/* dapm widget for Left Input Mixer*/
	SND_SOC_DAPM_MIXER("Left Input Mixer", SND_SOC_NOPM, 0, 0,
			   &left_input_mixer_controls[0],
			   ARRAY_SIZE(left_input_mixer_controls)),

	/* dapm widget for Right Input Mixer*/
	SND_SOC_DAPM_MIXER("Right Input Mixer", SND_SOC_NOPM, 0, 0,
			   &right_input_mixer_controls[0],
			   ARRAY_SIZE(right_input_mixer_controls)),

#ifndef AIC3253_CODEC_SUPPORT
	/* Left Inputs to Left ADC */
	SND_SOC_DAPM_ADC("Left ADC", "Left Capture", ADC_CHN_REG, 7, 0),

	/* Right Inputs to Right ADC */
	SND_SOC_DAPM_ADC("Right ADC", "Right Capture", ADC_CHN_REG, 6, 0),
#endif

	/* dapm widget (platform domain) name for HPLOUT */
	SND_SOC_DAPM_OUTPUT("HPL"),

	/* dapm widget (platform domain) name for HPROUT */
	SND_SOC_DAPM_OUTPUT("HPR"),

#ifndef AIC3253_CODEC_SUPPORT
	/* dapm widget (platform domain) name for LOLOUT */
	SND_SOC_DAPM_OUTPUT("LOL"),

	/* dapm widget (platform domain) name for LOROUT */
	SND_SOC_DAPM_OUTPUT("LOR"),

	/* dapm widget (platform domain) name for LINE1L */
	SND_SOC_DAPM_INPUT("IN1_L"),

	/* dapm widget (platform domain) name for LINE1R */
	SND_SOC_DAPM_INPUT("IN1_R"),

	/* dapm widget (platform domain) name for LINE2L */
	SND_SOC_DAPM_INPUT("IN2_L"),

	/* dapm widget (platform domain) name for LINE2R */
	SND_SOC_DAPM_INPUT("IN2_R"),

	/* dapm widget (platform domain) name for LINE3L */
	SND_SOC_DAPM_INPUT("IN3_L"),

	/* dapm widget (platform domain) name for LINE3R */
	SND_SOC_DAPM_INPUT("IN3_R"),
#endif

};

static const struct snd_soc_dapm_route aic32x4_dapm_routes[] = {

	/* Left  Headphone Output */
	{"HPL Output Mixer", "L_DAC switch", "Left DAC"},
	{"HPL Output Mixer", "IN1_L switch", "IN1_L"},
	//{"HPL Output Mixer", "Left_Bypass switch", "Left_Bypass"},

	{"HPL Power", NULL, "HPL Output Mixer"},
	{"HPL", NULL, "HPL Power"},

#ifndef AIC3253_CODEC_SUPPORT
	/* Left Line-out Output */
	{"LOL Output Mixer", "L_DAC switch", "Left DAC"},
//	{"LOL Output Mixer", "Left_Bypass switch", "Left_Bypass"},

	{"LOL Power", NULL, "LOL Output Mixer"},
	{"LOL", NULL, "LOL Power"},
#endif

	/* Right Headphone Output */
	{"HPR Output Mixer", "R_DAC switch", "Right DAC"},
	{"HPR Output Mixer", "IN1_R switch", "IN1_R"},
	//{"HPR Output Mixer", "Right_Bypass switch", "Right_Bypass"},

	{"HPR Power", NULL, "HPR Output Mixer"},
	{"HPR", NULL, "HPR Power"},

#ifndef AIC3253_CODEC_SUPPORT
	/* Right Line-out Output */
	{"LOR Output Mixer", "R_DAC switch", "Right DAC"},
//	{"LOR Output Mixer", "Right_Bypass switch", "Right_Bypass"},

	{"LOR Power", NULL, "LOR Output Mixer"},
	{"LOR", NULL, "LOR Power"},

	/* Left input */
	{"Left Input Mixer", "IN1_L switch", "IN1_L"},
	{"Left Input Mixer", "IN2_L switch", "IN2_L"},
	{"Left Input Mixer", "IN3_L switch", "IN3_L"},

	//{"Left_Bypass", NULL, "Left Input Mixer"},

	{"Left ADC", NULL, "Left Input Mixer"},

	/* Right Input */
	{"Right Input Mixer", "IN1_R switch", "IN1_R"},
	{"Right Input Mixer", "IN2_R switch", "IN2_R"},
	{"Right Input Mixer", "IN3_R switch", "IN3_R"},

//	{"Right_Bypass", NULL, "Right Input Mixer"},

	{"Right ADC", NULL, "Right Input Mixer"},
#endif
};

#define AIC32x4_DAPM_ROUTE_NUM (sizeof(aic32x4_dapm_routes)/sizeof(struct snd_soc_dapm_route))

//dwDBGain = ((dwGain*(DSP_PLAYBACK_GAIN_MAX_DB - DSP_PLAYBACK_GAIN_MIN_DB)))/100 + DSP_PLAYBACK_GAIN_MIN_DB;
//int dwLogGain = (int)(0x400000 * pow(10,(dwDBGain / 20)));
static int PlaybackPercentToLog[] =
{
	0xD,
	0xF,
	0x11,
	0x13,
	0x16,
	0x19,
	0x1D,
	0x21,
	0x26,
	0x2C,
	0x32,
	0x39,
	0x41,
	0x4B,
	0x56,
	0x62,
	0x70,
	0x80,
	0x92,
	0xA7,
	0xBF,
	0xDB,
	0xFA,
	0x11E,
	0x147,
	0x175,
	0x1AB,
	0x1E8,
	0x22E,
	0x27D,
	0x2D8,
	0x341,
	0x3B8,
	0x440,
	0x4DB,
	0x58D,
	0x658,
	0x740,
	0x849,
	0x978,
	0xAD3,
	0xC5F,
	0xE23,
	0x1028,
	0x1277,
	0x151B,
	0x181F,
	0x1B91,
	0x1F82,
	0x2402,
	0x2927,
	0x2F08,
	0x35C1,
	0x3D6F,
	0x4636,
	0x503E,
	0x5BB5,
	0x68D0,
	0x77CA,
	0x88E7,
	0x9C77,
	0xB2D2,
	0xCC5F,
	0xE992,
	0x10AF1,
	0x13115,
	0x15CAC,
	0x18E7E,
	0x1C76D,
	0x2087F,
	0x252DD,
	0x2A7DC,
	0x308FF,
	0x37803,
	0x3F6E4,
	0x487E5,
	0x52D9F,
	0x5EB06,
	0x6C37D,
	0x7BAE1,
	0x8D59F,
	0xA18C2,
	0xB8A10,
	0xD3022,
	0xF1283,
	0x1139CF,
	0x13AFE0,
	0x167FF7,
	0x19B6F0,
	0x1D6380,
	0x219672,
	0x2662FA,
	0x2BDF0A,
	0x3223B6,
	0x394DA8,
	0x417DA1,
	0x4AD913,
	0x558AC7,
	0x61C3A4,
	0x6FBB8C,
	0x7FB260,
};
//dwDBGain = ((dwGain*(DSP_RECORDING_GAIN_MAX_DB - DSP_RECORDING_GAIN_MIN_DB)))/200;
//int dwLogGain = (int)(0x400000 * pow(10,(dwDBGain / 20)));
static int RecordingPercentToLog[] =
{
	0x201373,
	0x204C5E,
	0x2085AF,
	0x20BF65,
	0x20F981,
	0x213405,
	0x216EF0,
	0x21AA44,
	0x21E601,
	0x222228,
	0x225EBA,
	0x229BB8,
	0x22D921,
	0x2316F8,
	0x23553C,
	0x2393EF,
	0x23D311,
	0x2412A3,
	0x2452A6,
	0x24931B,
	0x24D402,
	0x25155C,
	0x25572A,
	0x25996C,
	0x25DC25,
	0x261F54,
	0x2662FA,
	0x26A718,
	0x26EBAF,
	0x2730BF,
	0x27764A,
	0x27BC51,
	0x2802D4,
	0x2849D4,
	0x289152,
	0x28D94F,
	0x2921CB,
	0x296AC8,
	0x29B447,
	0x29FE48,
	0x2A48CD,
	0x2A93D5,
	0x2ADF63,
	0x2B2B77,
	0x2B7812,
	0x2BC535,
	0x2C12E1,
	0x2C6116,
	0x2CAFD6,
	0x2CFF23,
	0x2D4EFB,
	0x2D9F62,
	0x2DF057,
	0x2E41DC,
	0x2E93F1,
	0x2EE698,
	0x2F39D2,
	0x2F8DA0,
	0x2FE202,
	0x3036FA,
	0x308C88,
	0x30E2AF,
	0x31396E,
	0x3190C8,
	0x31E8BC,
	0x32414C,
	0x329A7A,
	0x32F446,
	0x334EB1,
	0x33A9BD,
	0x34056A,
	0x3461BA,
	0x34BEAD,
	0x351C46,
	0x357A85,
	0x35D96B,
	0x3638F9,
	0x369931,
	0x36FA13,
	0x375BA2,
	0x37BDDE,
	0x3820C8,
	0x388461,
	0x38E8AB,
	0x394DA8,
	0x39B357,
	0x3A19BB,
	0x3A80D4,
	0x3AE8A5,
	0x3B512E,
	0x3BBA70,
	0x3C246D,
	0x3C8F26,
	0x3CFA9C,
	0x3D66D1,
	0x3DD3C7,
	0x3E417D,
	0x3EAFF6,
	0x3F1F33,
	0x3F8F36,
	0x400000,
	0x407191,
	0x40E3EC,
	0x415712,
	0x41CB04,
	0x423FC4,
	0x42B554,
	0x432BB4,
	0x43A2E5,
	0x441AEB,
	0x4493C5,
	0x450D76,
	0x4587FF,
	0x460361,
	0x467F9E,
	0x46FCB8,
	0x477AAF,
	0x47F987,
	0x48793F,
	0x48F9DA,
	0x497B59,
	0x49FDBE,
	0x4A810A,
	0x4B053F,
	0x4B8A5F,
	0x4C106B,
	0x4C9765,
	0x4D1F4F,
	0x4DA829,
	0x4E31F7,
	0x4EBCB9,
	0x4F4871,
	0x4FD521,
	0x5062CB,
	0x50F170,
	0x518113,
	0x5211B4,
	0x52A356,
	0x5335FA,
	0x53C9A2,
	0x545E51,
	0x54F407,
	0x558AC7,
	0x562293,
	0x56BB6C,
	0x575554,
	0x57F04D,
	0x588C59,
	0x59297B,
	0x59C7B3,
	0x5A6703,
	0x5B076F,
	0x5BA8F7,
	0x5C4B9E,
	0x5CEF65,
	0x5D944F,
	0x5E3A5E,
	0x5EE193,
	0x5F89F1,
	0x60337A,
	0x60DE2F,
	0x618A14,
	0x62372A,
	0x62E572,
	0x6394F0,
	0x6445A6,
	0x64F795,
	0x65AABF,
	0x665F28,
	0x6714D1,
	0x67CBBC,
	0x6883EC,
	0x693D62,
	0x69F822,
	0x6AB42D,
	0x6B7186,
	0x6C302F,
	0x6CF02A,
	0x6DB179,
	0x6E7420,
	0x6F3820,
	0x6FFD7C,
	0x70C437,
	0x718C52,
	0x7255CF,
	0x7320B3,
	0x73ECFE,
	0x74BAB5,
	0x7589D8,
	0x765A6A,
	0x772C6F,
	0x77FFE8,
	0x78D4D9,
	0x79AB44,
	0x7A832B,
	0x7B5C91,
	0x7C3779,
	0x7D13E5,
	0x7DF1D9,
	0x7ED156,
	0x7FB260,
};

/*
*****************************************************************************
* Function Definitions
*****************************************************************************
*/
int ConvertDSPGain(int dwGain, int type)
{
	int dwLogGain = 0;

	if(type == PLAYBACK)
	{
		dwLogGain = PlaybackPercentToLog[dwGain];
	}
	else if(type == RECORDING)
	{
		dwLogGain = RecordingPercentToLog[dwGain-BOOST_MIN];
	}

	//shift to 0 to 0xFFFFFF00
	dwLogGain <<= 8;
	if(dwLogGain > 0x7FFFFFFF)
		dwLogGain = 0x7FFFFFFF;
	return dwLogGain;
}

static void SwitchAdaptivePage(struct snd_soc_codec *codec)
{
	u8 swapReg;

	swapReg = aic32x4_read(codec, DSP_ADC_ADAPTIVE_REG);
	aic32x4_write(codec, DSP_ADC_ADAPTIVE_REG, swapReg | DSP_ADC_ADAPTIVE_SWAP_BIT);
}

static void SetVolumes(struct snd_soc_codec *codec)
{
	u8 swapReg;
	int i;

	swapReg = aic32x4_read(codec, DSP_ADC_ADAPTIVE_REG);

	if(swapReg & DSP_ADC_ADAPTIVE_READ_BIT)//Buffer B in use, A free
	{
		for(i = 0; i < sizeof(int); i++)
		{
			aic32x4_write(codec, DSP_PLAYBACK_GAIN_A_REG + i, (uiAudio_data->play_vol_dsp >> 8*(3-i)) & 0xFF );
		}

		for(i = 0; i < sizeof(int); i++)
		{
			aic32x4_write(codec, DSP_RECORDING_GAIN_A_REG + i, (uiAudio_data->rec_boost_level_dsp >> 8*(3-i)) & 0xFF );
		}
	}
	else//Buffer A in use, B free
	{
		for(i = 0; i < sizeof(int); i++)
		{
			aic32x4_write(codec, DSP_PLAYBACK_GAIN_B_REG + i, (uiAudio_data->play_vol_dsp >> 8*(3-i)) & 0xFF );
		}

		for(i = 0; i < sizeof(int); i++)
		{
			aic32x4_write(codec, DSP_RECORDING_GAIN_B_REG + i, (uiAudio_data->rec_boost_level_dsp >> 8*(3-i)) & 0xFF );
		}
	}
	SwitchAdaptivePage(codec);
}

int playVolume(struct snd_soc_dai *dai)
{
    struct snd_soc_codec *codec = dai->codec;

	//mutex_lock(&codec->mutex); locked one level up
	/* do not do anything if play_capable is not set */
	if (uiAudio_data->play_capable == 0 || uiAudio_data->play_mute == 1)
		uiAudio_data->play_vol_dsp = 0;
	else
		uiAudio_data->play_vol_dsp = ConvertDSPGain(uiAudio_data->play_vol, PLAYBACK);

	SetVolumes(codec);
	//mutex_unlock(&codec->mutex); locked one level up
	return 0;

}
/* this function sets the mic gain(volume) */
int recBoostLevel(struct snd_soc_dai *dai)
{
	struct snd_soc_codec *codec = dai->codec;

	//mutex_lock(&codec->mutex);
//RJK  UI range is -100 - 100, but device shows range of 0 - 49db.

	/* do not do anything if play_capable is not set */
	if (uiAudio_data->rec_capable == 0 || uiAudio_data->rec_mute == 1)
		uiAudio_data->rec_boost_level_dsp = 0;
	else if(uiAudio_data->rec_boost_capable == 0)
		uiAudio_data->rec_boost_level_dsp=ConvertDSPGain(0, RECORDING);
	else
		uiAudio_data->rec_boost_level_dsp=ConvertDSPGain(uiAudio_data->rec_boost_level, RECORDING);
	SetVolumes(codec);

	//mutex_unlock(&codec->mutex);
	return 0;

}

/*
 *----------------------------------------------------------------------------
 * Function : snd_soc_info_volsw_2r_aic32x4
 * Purpose  : Callback to get the info of the value of a double mixer control
 * 	      that spans two registers.
 *----------------------------------------------------------------------------
 */
static int snd_soc_info_volsw_2r_aic32x4(struct snd_kcontrol *kcontrol,
                                         struct snd_ctl_elem_info *uinfo)
{
        int mask = (kcontrol->private_value >> 12) & 0xff;
        int min = (signed char)((kcontrol->private_value >> 24) & 0xff);
        int max = (signed char)((kcontrol->private_value >> 0) & 0xff);

        uinfo->type =
                mask == 1 ? SNDRV_CTL_ELEM_TYPE_BOOLEAN : SNDRV_CTL_ELEM_TYPE_INTEGER;
        uinfo->count = 2;
        uinfo->value.integer.min = min;
        uinfo->value.integer.max = max;
        return 0;
}

/*
 *----------------------------------------------------------------------------
 * Function : snd_soc_get_volsw_2r_aic32x4
 * Purpose  : Callback to get the value of a double mixer control that spans
 *            two registers.
 *----------------------------------------------------------------------------
 */
int snd_soc_get_volsw_2r_aic32x4(struct snd_kcontrol *kcontrol,
				 struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_codec *codec = snd_kcontrol_chip(kcontrol);
	int mask;
	int shift;
	unsigned short val, val2;

	mutex_lock(&codec->mutex);
	if (strcmp (kcontrol->id.name, "Left/Right DAC Digital Volume Control") == 0)
	{
		ucontrol->value.integer.value[0] = uiAudio_data->play_vol;
		ucontrol->value.integer.value[1] = uiAudio_data->play_vol;
	}

#ifndef AIC3253_CODEC_SUPPORT
	else if (strcmp (kcontrol->id.name, "Left/Right ADC Volume Control") == 0)
	{
		ucontrol->value.integer.value[0] = uiAudio_data->rec_boost_level;
		ucontrol->value.integer.value[1] = uiAudio_data->rec_boost_level;
	}
#endif
#ifdef AIC3253_CODEC_SUPPORT
	else if ((!strcmp(kcontrol->id.name, "HP Driver Gain"))
				 || (!strcmp(kcontrol->id.name, "LO Driver Gain"))) {
		ucontrol->value.integer.value[0] =
						(val <= 29) ? (val + 6) : (val - 58);
		ucontrol->value.integer.value[1] =
						(val2 <= 29) ? (val2 + 6) : (val2 - 58);
	}
#endif
	mutex_unlock(&codec->mutex);
    return 0;
}


/*
 *----------------------------------------------------------------------------
 * Function : snd_soc_put_volsw_2r_aic32x4
 * Purpose  : Callback to set the value of a double mixer control that spans
 *            two registers.
 *----------------------------------------------------------------------------
 */
int snd_soc_put_volsw_2r_aic32x4(struct snd_kcontrol *kcontrol,
                                 struct snd_ctl_elem_value *ucontrol)
{
        struct snd_soc_codec *codec = snd_kcontrol_chip(kcontrol);
        int err;
        unsigned short val, val2, val_mask;

        //only use left channel since we arent adjusting volume separately
        val = ucontrol->value.integer.value[0];
        val2 = ucontrol->value.integer.value[1];

    	mutex_lock(&codec->mutex);
    	if (strcmp (kcontrol->id.name, "Left/Right DAC Digital Volume Control") == 0) {
        	 uiAudio_data->play_vol = val;
        	 playVolume(codec->dai);

		}

        #ifndef AIC3253_CODEC_SUPPORT
        else if (strcmp (kcontrol->id.name, "Left/Right ADC Volume Control") == 0) {
        	 uiAudio_data->rec_boost_level = val;
        	 recBoostLevel(codec->dai);
		}
#if 0 //unused on crestron products
        else if ((!strcmp(kcontrol->id.name, "HP Driver Gain")) ||
				   (!strcmp(kcontrol->id.name, "LO Driver Gain"))) {
				val = (val >= 6) ? (val - 6) : (val + 58);
				val2 = (val2 >= 6) ? (val2 - 6) : (val2 + 58);
				val_mask = 0x3F;	/* 6 bits */
		}
#endif
        #endif

        #ifdef AIC3253_CODEC_SUPPORT
		else if((!strcmp(kcontrol->id.name, "HP Driver Gain"))) {
				val = (val >= 6) ? (val - 6) : (val + 58);
				val2 = (val2 >= 6) ? (val2 - 6) : (val2 + 58);
				val_mask = 0x3F;	/* 6 bits */
		}
        #endif
    	mutex_unlock(&codec->mutex);
        return err;
}

/*
 *----------------------------------------------------------------------------
 * Function : __new_control_info
 * Purpose  : This function is to initialize data for new control required to
 *            program the AIC32x4 registers.
 *----------------------------------------------------------------------------
 */
static int __new_control_info(struct snd_kcontrol *kcontrol,
                              struct snd_ctl_elem_info *uinfo)
{
        uinfo->type = SNDRV_CTL_ELEM_TYPE_INTEGER;
        uinfo->count = 1;
        uinfo->value.integer.min = 0;
        uinfo->value.integer.max = 65535;

        return 0;
}

/*
 *----------------------------------------------------------------------------
 * Function : __new_control_get
 * Purpose  : This function is to read data of new control for
 *            program the AIC32x4 registers.
 *----------------------------------------------------------------------------
 */
static int __new_control_get(struct snd_kcontrol *kcontrol,
                             struct snd_ctl_elem_value *ucontrol)
{
        struct snd_soc_codec *codec = snd_kcontrol_chip(kcontrol);
        u32 val;

        val = aic32x4_read(codec, aic32x4_reg_ctl);
        ucontrol->value.integer.value[0] = val;

        return 0;
}

/*
 *----------------------------------------------------------------------------
 * Function : __new_control_put
 * Purpose  : new_control_put is called to pass data from user/application to
 *            the driver.
 *----------------------------------------------------------------------------
 */
static int __new_control_put(struct snd_kcontrol *kcontrol,
                             struct snd_ctl_elem_value *ucontrol)
{
        struct snd_soc_codec *codec = snd_kcontrol_chip(kcontrol);
        struct aic32x4_priv *aic32x4 = codec->private_data;

        u32 data_from_user = ucontrol->value.integer.value[0];
        u8 data[2];

        aic32x4_reg_ctl = data[0] = (u8) ((data_from_user & 0xFF00) >> 8);
        data[1] = (u8) ((data_from_user & 0x00FF));

        if (!data[0]) {
                aic32x4->page_no = data[1];
        }

        printk("reg = %d val = %x\n", data[0], data[1]);

        if (codec->hw_write(codec->control_data, data, 2) != 2) {
                printk("Func: __new_control_put, Error in i2c write\n");
                return -EIO;
        }
        return 0;
}

/*
 *----------------------------------------------------------------------------
 * Function : aic32x4_reset_cache
 * Purpose  : This function is to reset the cache.
 *----------------------------------------------------------------------------
 */
int aic32x4_reset_cache (struct snd_soc_codec *codec)
{
#if defined(EN_REG_CACHE)
        if (codec->reg_cache != NULL)
        {
                memcpy(codec->reg_cache, aic32x4_reg, sizeof (aic32x4_reg));
                return 0;
        }

        codec->reg_cache = kmemdup (aic32x4_reg, sizeof (aic32x4_reg), GFP_KERNEL);
        if (codec->reg_cache == NULL)
        {
                printk (KERN_ERR "aic32x4: kmemdup failed\n");
                return -ENOMEM;
        }
#endif
        return 0;
}


/*
 *----------------------------------------------------------------------------
 * Function : aic32x4_change_page
 * Purpose  : This function is to switch between page 0 and page 1.
 *----------------------------------------------------------------------------
 */
int aic32x4_change_page(struct snd_soc_codec *codec, u8 new_page)
{
        struct aic32x4_priv *aic32x4 = codec->private_data;
        u8 data[2];

        data[0] = 0;
        data[1] = new_page;
        aic32x4->page_no = new_page;

        if (codec->hw_write(codec->control_data, data, 2) != 2) {
                printk("Func:aic32x4_change_page, Error in changing page \n");
                return -1;
        }
        return 0;
}

/*
 *----------------------------------------------------------------------------
 * Function : aic32x4_write_reg_cache
 * Purpose  : This function is to write aic32x4 register cache
 *----------------------------------------------------------------------------
 */
inline void aic32x4_write_reg_cache(struct snd_soc_codec *codec,
                                    u16 reg, u8 value)
{
        u8 *cache = codec->reg_cache;

        if (reg >= AIC32x4_CACHEREGNUM) {
                return;
        }
        cache[reg] = value;
}

/*
 *----------------------------------------------------------------------------
 * Function : aic32x4_write
 * Purpose  : This function is to write to the aic32x4 register space.
 *----------------------------------------------------------------------------
 */
int aic32x4_write(struct snd_soc_codec *codec, u16 reg, u8 value)
{
        struct aic32x4_priv *aic32x4 = codec->private_data;
        u8 data[2];
        u8 page;

        page = reg / 128;
        data[AIC32x4_REG_OFFSET_INDEX] = reg % 128;

        if (aic32x4->page_no != page) {
                aic32x4_change_page(codec, page);
        }


        /* data is
         *   D15..D8 aic32x4 register offset
         *   D7...D0 register data
         */
        data[AIC32x4_REG_DATA_INDEX] = value & AIC32x4_8BITS_MASK;

#if defined(EN_REG_CACHE)
        if ((page == 0) || (page == 1)) {
                aic32x4_write_reg_cache(codec, reg, value);
        }
#endif
        if (!data[AIC32x4_REG_OFFSET_INDEX]) {
                /* if the write is to reg0 update aic32x4->page_no */
                aic32x4->page_no = value;
        }

	dprintk("page->%d\t register->%d\t value->%d\r\n",
                page, (reg %128), value);

        if (codec->hw_write(codec->control_data, data, 2) != 2) {
                printk("Func:aic32x4_write, Error in i2c write\n");
                return -EIO;
        }


        return 0;
}

/*
 *----------------------------------------------------------------------------
 * Function : aic32x4_read
 * Purpose  : This function is to read the aic32x4 register space.
 *----------------------------------------------------------------------------
 */
u8 aic32x4_read(struct snd_soc_codec *codec, u16 reg)
{
        struct aic32x4_priv *aic32x4 = codec->private_data;
        u8 value;
        u8 page = reg / 128;

        reg = reg % 128;

        if (aic32x4->page_no != page) {
                aic32x4_change_page(codec, page);
        }

        i2c_master_send(codec->control_data, (char *)&reg, 1);
        i2c_master_recv(codec->control_data, &value, 1);
        return value;
}

/*
 *----------------------------------------------------------------------------
 * Function : aic32x4_get_divs
 * Purpose  : This function is to get required divisor from the "aic32x4_divs"
 *            table.
 *----------------------------------------------------------------------------
 */
static inline int aic32x4_get_divs(int mclk, int rate)
{
        int i;
#ifdef RJK_DEBUG_FS
	if (g_dbg_rate) {
		rate=g_dbg_rate;	
	}
#endif

        for (i = 0; i < ARRAY_SIZE(aic32x4_divs); i++) {
                if ((aic32x4_divs[i].rate == rate)
                    && (aic32x4_divs[i].mclk == mclk)) {
printk(KERN_ERR "Setting sample rate to element=%d\n", i);
                        return i;
                }
        }

        printk("Master clock and sample rate is not supported\n");
        return -EINVAL;
}

/*
 *----------------------------------------------------------------------------
 * Function : aic32x4_add_widgets
 * Purpose  : This function is to add the dapm widgets. This routine will be
 *	      invoked during the Audio Driver Initialization.
 *            The following are the main widgets supported :
 *                # Left DAC to Left Outputs
 *                # Right DAC to Right Outputs
 *		  # Left Inputs to Left ADC
 *		  # Right Inputs to Right ADC
 *----------------------------------------------------------------------------
 */
static int aic32x4_add_widgets(struct snd_soc_codec *codec)
{
        int i;

	dprintk( KERN_INFO "Function : %s \n", __FUNCTION__);
        for (i = 0; i < ARRAY_SIZE(aic32x4_dapm_widgets); i++) {
                snd_soc_dapm_new_control(codec, &aic32x4_dapm_widgets[i]);
        }

        /* set up audi path interconnects */
        snd_soc_dapm_add_routes(codec, &aic32x4_dapm_routes[0],
                                AIC32x4_DAPM_ROUTE_NUM);

        snd_soc_dapm_new_widgets(codec);
        return 0;
}

/*
 *----------------------------------------------------------------------------
 * Function : aic32x4_hw_params
 * Purpose  : This function is to set the hardware parameters for AIC32x4.
 *            The functions set the sample rate and audio serial data word
 *            length.
 *----------------------------------------------------------------------------
 */
static int aic32x4_hw_params(struct snd_pcm_substream *substream,
                             struct snd_pcm_hw_params *params,
                             struct snd_soc_dai *dai)
{
        struct snd_soc_pcm_runtime *rtd = substream->private_data;
        struct snd_soc_device *socdev = rtd->socdev;
        struct snd_soc_codec *codec = socdev->card->codec;
        struct aic32x4_priv *aic32x4 = codec->private_data;
        int i, j;
        u8 data;
	u32 rate=0;

	dprintk( KERN_INFO "Function : %s \n", __FUNCTION__);
        aic32x4_set_bias_level(codec, SNDRV_CTL_POWER_D3hot);

        i = aic32x4_get_divs(aic32x4->sysclk, params_rate(params));
	rate = params_rate(params);
//printk(KERN_ERR "RJK: in aic32x4_hw_params rate=%d \n", rate);

        if (i < 0) {
                printk("sampling rate not supported\n");
                return i;
        }

        if (soc_static_freq_config) {
                /* We will fix R value to 1 and will make P & J=K.D as varialble */
//printk(KERN_ERR "RJK: in aic32x4_hw_params aic32x4_divs[%d] element\n", i);
 
                /* Setting P & R values */
                aic32x4_write(codec, CLK_REG_2,
                              ((aic32x4_divs[i].p_val << 4) | 0x01));

                /* J value */
                aic32x4_write(codec, CLK_REG_3, aic32x4_divs[i].pll_j);

                /* MSB & LSB for D value */
                aic32x4_write(codec, CLK_REG_4, (aic32x4_divs[i].pll_d >> 8));
                aic32x4_write(codec, CLK_REG_5,
                              (aic32x4_divs[i].pll_d & AIC32x4_8BITS_MASK));

                /* NDAC divider value */
                aic32x4_write(codec, NDAC_CLK_REG_6, aic32x4_divs[i].ndac);

                /* MDAC divider value */
                aic32x4_write(codec, MDAC_CLK_REG_7, aic32x4_divs[i].mdac);

                /* DOSR MSB & LSB values */
                aic32x4_write(codec, DAC_OSR_MSB, aic32x4_divs[i].dosr >> 8);
                aic32x4_write(codec, DAC_OSR_LSB,
                              aic32x4_divs[i].dosr & AIC32x4_8BITS_MASK);

#ifndef AIC3253_CODEC_SUPPORT
                /* NADC divider value */
                aic32x4_write(codec, NADC_CLK_REG_8, aic32x4_divs[i].nadc);

                /* MADC divider value */
                aic32x4_write(codec, MADC_CLK_REG_9, aic32x4_divs[i].madc);
#endif

                /* AOSR value */
                aic32x4_write(codec, ADC_OSR_REG, aic32x4_divs[i].aosr);
        }
        /* BCLK N divider */
        aic32x4_write(codec, CLK_REG_11, aic32x4_divs[i].blck_N);

        aic32x4_set_bias_level(codec, SNDRV_CTL_POWER_D0);

        data = aic32x4_read(codec, INTERFACE_SET_REG_1);

        data = data & ~(3 << 4);

        switch (params_format(params)) {
        case SNDRV_PCM_FORMAT_S16_LE:
                break;
        case SNDRV_PCM_FORMAT_S20_3LE:
                data |= (AIC32x4_WORD_LEN_20BITS << DAC_OSR_MSB_SHIFT);
                break;
        case SNDRV_PCM_FORMAT_S24_LE:
                data |= (AIC32x4_WORD_LEN_24BITS << DAC_OSR_MSB_SHIFT);
                break;
        case SNDRV_PCM_FORMAT_S32_LE:
                data |= (AIC32x4_WORD_LEN_32BITS << DAC_OSR_MSB_SHIFT);
                break;
        }

        aic32x4_write(codec, INTERFACE_SET_REG_1, data);

        for (j = 0; j < NO_FEATURE_REGS; j++) {
                aic32x4_write(codec,
                              aic32x4_divs[i].codec_specific_regs[j].reg_offset,
                              aic32x4_divs[i].codec_specific_regs[j].reg_val);
        }
        return 0;
}

/*
 *----------------------------------------------------------------------
 * Function : aic32x4_config_volume
 * Purpose  : This function is used to configure the I2C Transaction
 *	      Global Variables. One of them is for ramping down the HP
 *	      Analog Volume and the other one is for ramping up the HP 
 *	      Analog Volume

*/ 

/*
 *-----------------------------------------------------------------------
 * Function : aic32x4_mute
 * Purpose  : This function is to mute or unmute the left and right DAC
 *-----------------------------------------------------------------------
 */
//static int aic32x4_mute(struct snd_soc_dai *dai, int mute)
int aic32x4_mute(struct snd_soc_dai *dai, int mute)
{
        struct snd_soc_codec *codec = dai->codec;
        u8 dac_reg;
		u16 counter;
		
	//printk( KERN_ERR "RJK: Function : %s called with mute=0x%x\n", __FUNCTION__,mute);
	dprintk( KERN_INFO "Function : %s called with mute=0x%x\n", __FUNCTION__,mute);
        dac_reg = aic32x4_read(codec, DAC_MUTE_CTRL_REG) & ~MUTE_ON;
        if (mute)
                aic32x4_write(codec, DAC_MUTE_CTRL_REG, dac_reg | MUTE_ON);
        else {
                aic32x4_write(codec, DAC_MUTE_CTRL_REG, dac_reg);
		#if 0 //DEBUG		
				for(counter = 0; counter < 256; counter++) {
					printk (KERN_ERR "Pg%d Reg %d Val 0x%x\n",
								(counter/128), (counter%128), aic32x4_read(codec, counter));
				}
		#endif
		}
        return 0;
}

/*
 *----------------------------------------------------------------------------
 * Function : aic32x4_set_dai_sysclk
 * Purpose  : This function is to set the DAI system clock
 *----------------------------------------------------------------------------
 */
static int aic32x4_set_dai_sysclk(struct snd_soc_dai *codec_dai,
                                  int clk_id, unsigned int freq, int dir)
{
        struct snd_soc_codec *codec = codec_dai->codec;
        struct aic32x4_priv *aic32x4 = codec->private_data;

	dprintk( KERN_INFO "Function : %s \n", __FUNCTION__);
	//printk(KERN_ERR "RJK:: %s audio freq =0x%x-%d-\n", codec_dai->name, freq, freq);

	if (clk_id == SGTL5000_SYSCLK) {
        	switch (freq) {
//        	case AIC32x4_FREQ_12000000:
        	case AIC32x4_FREQ_24000000:
                	aic32x4->sysclk = freq;
                	return 0;

		default:  //RJK added for debug, remove this default case..... 
        		printk("Invalid frequency to set DAI system clock forcing to 24000000\n");
                	aic32x4->sysclk = AIC32x4_FREQ_24000000;
                	return 0;
		
        	}
	}
	if (clk_id == SGTL5000_LRCLK) {
		//RJK: May need to do something here for DSP not working from android
		return 0;
	}

        return -EINVAL;
}

/*
 *----------------------------------------------------------------------------
 * Function : aic32x4_set_dai_fmt
 * Purpose  : This function is to set the DAI format
 *----------------------------------------------------------------------------
 */
static int aic32x4_set_dai_fmt(struct snd_soc_dai *codec_dai, unsigned int fmt)
{
        struct snd_soc_codec *codec = codec_dai->codec;
        struct aic32x4_priv *aic32x4 = codec->private_data;
        u8 iface_reg;
	u8 iface_reg1;
	
	dprintk( KERN_INFO "Function : %s \n", __FUNCTION__);
        iface_reg = aic32x4_read(codec, INTERFACE_SET_REG_1);
        iface_reg = iface_reg & ~(3 << 6 | 3 << 2);

        /* set master/slave audio interface */
        switch (fmt & SND_SOC_DAIFMT_MASTER_MASK) {
        case SND_SOC_DAIFMT_CBM_CFM:
                aic32x4->master = 1;
                iface_reg |= BIT_CLK_MASTER | WORD_CLK_MASTER;
                break;
        case SND_SOC_DAIFMT_CBS_CFS:
                aic32x4->master = 0;
		iface_reg1 &= ~(BIT_CLK_MASTER | WORD_CLK_MASTER);
                break;
	case SND_SOC_DAIFMT_CBS_CFM:
		aic32x4->master = 0;
		iface_reg1 |= BIT_CLK_MASTER;
		iface_reg1 &= ~(WORD_CLK_MASTER);
		break;
        default:
                printk("Invalid DAI master/slave interface\n");
                return -EINVAL;
        }

        /* interface format */
        switch (fmt & SND_SOC_DAIFMT_FORMAT_MASK) {
        case SND_SOC_DAIFMT_I2S:
                break;
        case SND_SOC_DAIFMT_DSP_A:
                iface_reg |= (AIC32x4_DSP_MODE << CLK_REG_3_SHIFT);
                break;
        case SND_SOC_DAIFMT_RIGHT_J:
                iface_reg |= (AIC32x4_RIGHT_JUSTIFIED_MODE << CLK_REG_3_SHIFT);
                break;
        case SND_SOC_DAIFMT_LEFT_J:
                iface_reg |= (AIC32x4_LEFT_JUSTIFIED_MODE << CLK_REG_3_SHIFT);
                break;
        default:
                printk("Invalid DAI interface format\n");
                return -EINVAL;
        }

        aic32x4_write(codec, INTERFACE_SET_REG_1, iface_reg);
        return 0;
}

/*
 *----------------------------------------------------------------------------
 * Function : aic32x4_set_bias_level
 * Purpose  : This function is to get triggered when dapm events occurs.
 *----------------------------------------------------------------------------
 */
static int aic32x4_set_bias_level(struct snd_soc_codec *codec,
                                  enum snd_soc_bias_level level)
{
        struct aic32x4_priv *aic32x4 = codec->private_data;
        u8 value;

	dprintk( KERN_INFO "Function : %s \n", __FUNCTION__);
        switch (level) {
                /* full On */
        case SND_SOC_BIAS_ON:

                dprintk( KERN_INFO "Function : %s: BIAS ON \n", __FUNCTION__);
                /* all power is driven by DAPM system */
                if (aic32x4->master) {
                        /* Switch on PLL */
                        value = aic32x4_read(codec, CLK_REG_2);
                        aic32x4_write(codec, CLK_REG_2, (value | ENABLE_PLL));

                        /* Switch on NDAC Divider */
                        value = aic32x4_read(codec, NDAC_CLK_REG_6);
                        aic32x4_write(codec, NDAC_CLK_REG_6,
                                      value | ENABLE_NDAC);

                        /* Switch on MDAC Divider */
                        value = aic32x4_read(codec, MDAC_CLK_REG_7);
                        aic32x4_write(codec, MDAC_CLK_REG_7,
                                      value | ENABLE_MDAC);
#ifndef AIC3253_CODEC_SUPPORT

                        /* Switch on NADC Divider */
                        value = aic32x4_read(codec, NADC_CLK_REG_8);
                        aic32x4_write(codec, NADC_CLK_REG_8,
                                      value | ENABLE_NADC);

                        /* Switch on MADC Divider */
                        value = aic32x4_read(codec, MADC_CLK_REG_9);
                        aic32x4_write(codec, MADC_CLK_REG_9,
                                      value | ENABLE_MADC);

#endif
                        /* Switch on BCLK_N Divider */
                        value = aic32x4_read(codec, CLK_REG_11);
                        aic32x4_write(codec, CLK_REG_11, value | ENABLE_BCLK);
                }
                break;

                /* partial On */
        case SND_SOC_BIAS_PREPARE:
                dprintk( KERN_INFO "Function : %s: BIAS PREPARE \n", __FUNCTION__);
                break;

                /* Off, with power */
        case SND_SOC_BIAS_STANDBY:
                dprintk( KERN_INFO "Function : %s: BIAS STANDBY\n", __FUNCTION__);
                /*
                 * all power is driven by DAPM system,
                 * so output power is safe if bypass was set
                 */
                if (aic32x4->master) {
                      
                        /* Switch off NDAC Divider */
                        value = aic32x4_read(codec, NDAC_CLK_REG_6);
                        aic32x4_write(codec, NDAC_CLK_REG_6,
                                      value & ~ENABLE_NDAC);

                        /* Switch off MDAC Divider */
                        value = aic32x4_read(codec, MDAC_CLK_REG_7);
                        aic32x4_write(codec, MDAC_CLK_REG_7,
                                      value & ~ENABLE_MDAC);
#ifndef AIC3253_CODEC_SUPPORT

                        /* Switch off NADC Divider */
                        value = aic32x4_read(codec, NADC_CLK_REG_8);
                        aic32x4_write(codec, NADC_CLK_REG_8,
                                      value & ~ENABLE_NDAC);

                        /* Switch off MADC Divider */
                        value = aic32x4_read(codec, MADC_CLK_REG_9);
                        aic32x4_write(codec, MADC_CLK_REG_9,
                                      value & ~ENABLE_MDAC);
                        value = aic32x4_read(codec, CLK_REG_11);

#endif

                        /* Switch off BCLK_N Divider */
                        aic32x4_write(codec, CLK_REG_11, value & ~ENABLE_BCLK);

			  /* Switch off PLL */
                        value = aic32x4_read(codec, CLK_REG_2);
                        aic32x4_write(codec, CLK_REG_2, (value & ~ENABLE_PLL));

                }
                break;

                /* Off, without power */
        case SND_SOC_BIAS_OFF:
                dprintk( KERN_INFO "Function : %s: BIAS OFF \n", __FUNCTION__);
                /* force all power off */
                break;
        }
        codec->bias_level = level;
        return 0;
}

/*
 *----------------------------------------------------------------------------
 * Function : aic32x4_suspend
 * Purpose  : This function is to suspend the AIC32x4 driver.
 *----------------------------------------------------------------------------
 */
static int aic32x4_suspend(struct platform_device *pdev, pm_message_t state)
{
        struct snd_soc_device *socdev = platform_get_drvdata(pdev);
        struct snd_soc_codec *codec = socdev->card->codec;

	dprintk( KERN_INFO "Function : %s \n", __FUNCTION__);
        aic32x4_set_bias_level(codec, SNDRV_CTL_POWER_D3cold);

        return 0;
}

/*
 *----------------------------------------------------------------------------
 * Function : aic32x4_resume
 * Purpose  : This function is to resume the AIC32x4 driver
 *----------------------------------------------------------------------------
 */
static int aic32x4_resume(struct platform_device *pdev)
{
        struct snd_soc_device *socdev = platform_get_drvdata(pdev);
        struct snd_soc_codec *codec = socdev->card->codec;
        int i;
        u8 data[2];
        u8 *cache = codec->reg_cache;

	dprintk( KERN_INFO "Function : %s \n", __FUNCTION__);
        aic32x4_change_page(codec, 0);
        /* Sync reg_cache with the hardware */
        for (i = 0; i < ARRAY_SIZE(aic32x4_reg); i++) {
                data[0] = i % 128;
                data[1] = cache[i];
                codec->hw_write(codec->control_data, data, 2);
        }
        aic32x4_change_page(codec, 0);
        aic32x4_set_bias_level(codec, codec->suspend_bias_level);

        return 0;
}

/*
 *----------------------------------------------------------------------------
 * Function : aic32x4_init
 * Purpose  : This function is to initialise the AIC32x4 driver
 *            register the mixer and dsp interfaces with the kernel.
 *
 *----------------------------------------------------------------------------
 */
static int tlv320aic32x4_init(struct snd_soc_device *socdev)
{
        struct snd_soc_codec *codec = socdev->card->codec;
        struct aic32x4_priv *aic32x4 = codec->private_data;
        int ret = 0;
        int i = 0;
        aic32x4->page_no = 0;

	dprintk( KERN_INFO "Function : %s \n", __FUNCTION__);

	/* register pcms */
        ret = snd_soc_new_pcms(socdev, SNDRV_DEFAULT_IDX1, SNDRV_DEFAULT_STR1);
        if (ret < 0) {
                printk(KERN_ERR "aic32x4: failed to create pcms\n");
                goto pcm_err;
        }

        for (i = 0;
             i < sizeof(aic32x4_reg_init) / sizeof(struct aic32x4_configs);
             i++) {
                aic32x4_write(codec, aic32x4_reg_init[i].reg_offset,
                              aic32x4_reg_init[i].reg_val);
        }

        /* off, with power on */
        aic32x4_set_bias_level(codec, SNDRV_CTL_POWER_D3hot);
	snd_soc_add_controls(codec, aic32x4_snd_controls1,
                             ARRAY_SIZE(aic32x4_snd_controls1));
	snd_soc_add_controls(codec, aic32x4_snd_controls2,
                             ARRAY_SIZE(aic32x4_snd_controls2));
	snd_soc_add_controls(codec, aic32x4_snd_controls3,
                             ARRAY_SIZE(aic32x4_snd_controls3));
	snd_soc_add_controls(codec, aic32x4_snd_controls4,
                             ARRAY_SIZE(aic32x4_snd_controls4));
        aic32x4_add_widgets(codec);

        aic32x4_add_minidsp_controls(codec);

#ifdef AIC3xxx_TiLoad
        aic3xxx_driver_init(codec);
#endif

        return ret;

pcm_err:
        kfree(codec->reg_cache);
        return ret;
}

#if defined(CONFIG_I2C) || defined(CONFIG_I2C_MODULE)
/*
 *----------------------------------------------------------------------------
 * Function : aic32x4_codec_probe
 * Purpose  : This function attaches the i2c client and initializes
 *				AIC32x4 CODEC.
 *            NOTE:
 *            This function is called from i2c core when the I2C address is
 *            valid.
 *            If the i2c layer weren't so broken, we could pass this kind of
 *            data around
 *----------------------------------------------------------------------------
 */
static int tlv320aic32x4_codec_probe(struct i2c_client *i2c, const struct i2c_device_id *id)
{
        struct snd_soc_device *socdev = aic32x4_socdev;
        struct snd_soc_codec *codec = socdev->card->codec;
        int ret;

	dprintk( KERN_INFO "Function : %s \n", __FUNCTION__);
        i2c_set_clientdata(i2c, codec);
        codec->control_data = i2c;

        ret = tlv320aic32x4_init(socdev);
        if (ret < 0) {
                printk(KERN_ERR "aic32x4: failed to attach codec at addr\n");
                return -1;
        }
        /* Program MINI DSP for ADC and DAC */
        aic32x4_minidsp_program(codec);
        aic32x4_change_page(codec, 0x0);

        return ret;
}

/*
 *----------------------------------------------------------------------------
 * Function : tlv320aic32x4_i2c_remove
 * Purpose  : This function removes the i2c client and uninitializes
 *                              AIC32x4 CODEC.
 *            NOTE:
 *            This function is called from i2c core
 *            If the i2c layer weren't so broken, we could pass this kind of
 *            data around
 *----------------------------------------------------------------------------
 */
static int __exit tlv320aic32x4_i2c_remove(struct i2c_client *i2c)
{
	dprintk( KERN_INFO "Function : %s \n", __FUNCTION__);
        put_device(&i2c->dev);
        return 0;
}

static const struct i2c_device_id tlv320aic32x4_id[] = {
        {"tlv320aic32x4", 0},
        {}
};

MODULE_DEVICE_TABLE(i2c, tlv320aic32x4_id);

static struct i2c_driver tlv320aic32x4_i2c_driver = {
        .driver = {
                .name = "tlv320aic32x4",
        },
        .probe = tlv320aic32x4_codec_probe,
        .remove = __exit_p(tlv320aic32x4_i2c_remove),
        .id_table = tlv320aic32x4_id,
};

#endif //#if defined(CONFIG_I2C) || defined(CONFIG_I2C_MODULE)

/*
 *----------------------------------------------------------------------------
 * Function : aic32x4_hw_read
 * Purpose  : i2c read function
 *----------------------------------------------------------------------------
 */
unsigned int
aic32x4_hw_read (struct snd_soc_codec *codec, unsigned int count)
{
        struct i2c_client *client = codec->control_data;
        unsigned int buf;

        if (count > sizeof (unsigned int)) {
                return 0;
        }
        i2c_master_recv(client, (char *)&buf, count);
        return buf;
}

/*
 *----------------------------------------------------------------------------
 * Function : aic32x4_probe
 * Purpose  : This is first driver function called by the SoC core driver.
 *
 *----------------------------------------------------------------------------
 */
static int
aic32x4_probe(struct platform_device *pdev)
{
        struct snd_soc_device *socdev = platform_get_drvdata(pdev);
        struct snd_soc_codec *codec;
        struct aic32x4_priv *aic32x4;
        int ret = 0;
	
        codec = kzalloc(sizeof(struct snd_soc_codec), GFP_KERNEL);
        if(codec == NULL)
                return -ENOMEM;

	aic32x4 = kzalloc(sizeof(struct aic32x4_priv), GFP_KERNEL);
        if (aic32x4 == NULL) {
                kfree(codec);
                return -ENOMEM;
        }

	socdev->card->codec = codec;
        codec->private_data = aic32x4;
        codec->dev = &pdev->dev;

        mutex_init(&codec->mutex);
        INIT_LIST_HEAD(&codec->dapm_widgets);
        INIT_LIST_HEAD(&codec->dapm_paths);

        codec->name = "aic32x4";
        codec->owner = THIS_MODULE;
        codec->read = aic32x4_read;
        codec->write = aic32x4_write;
        codec->set_bias_level = aic32x4_set_bias_level;
        codec->dai = &tlv320aic32x4_dai;
        codec->num_dai = 1;
        codec->reg_cache_size = sizeof(aic32x4_reg);
        codec->reg_cache =
                kmemdup(aic32x4_reg, sizeof(aic32x4_reg), GFP_KERNEL);

        if (codec->reg_cache == NULL) {
                printk(KERN_ERR "aic32x4: kmemdup failed\n");
                return -ENOMEM;
        }

        aic32x4_socdev = socdev;
#if defined(CONFIG_I2C) || defined(CONFIG_I2C_MODULE)
        codec->hw_write = (hw_write_t) i2c_master_send;
        codec->hw_read = aic32x4_hw_read;

        ret = i2c_add_driver(&tlv320aic32x4_i2c_driver);
        if (ret != 0)
                printk(KERN_ERR "can't add i2c driver");
	else
		printk(KERN_ERR "Added i2c driver for aic32x4\n");

	//RJK setting up debugfs
	init_audio_debugfs(codec);

	uiAudioInit(codec);

#else
        /* Add other interfaces here */
#endif
	return ret;
}

/*
 *----------------------------------------------------------------------------
 * Function : aic32x4_remove
 * Purpose  : to remove aic32x4 soc device
 *
 *----------------------------------------------------------------------------
 */
static int
aic32x4_remove(struct platform_device *pdev)
{
        struct snd_soc_device *socdev = platform_get_drvdata(pdev);
        struct snd_soc_codec *codec = socdev->card->codec;

	dprintk( KERN_INFO "Function : %s \n", __FUNCTION__);

        /* power down chip */
        if (codec->control_data)
                aic32x4_set_bias_level(codec, SNDRV_CTL_POWER_D3);

        snd_soc_free_pcms(socdev);
        snd_soc_dapm_free(socdev);
#if defined(CONFIG_I2C) || defined(CONFIG_I2C_MODULE)
        i2c_del_driver(&tlv320aic32x4_i2c_driver);
#endif
        kfree(codec->private_data);
        kfree(codec);

        return 0;
}

/*
 *----------------------------------------------------------------------------
 * @struct  snd_soc_codec_device |
 *          This structure is soc audio codec device sturecute which pointer
 *          to basic functions aic32x4_probe(), aic32x4_remove(),
 *          aic32x4_suspend() and aic32x4_resume()
 *----------------------------------------------------------------------------
 */
struct snd_soc_codec_device soc_codec_dev_aic32x4 = {
        .probe = aic32x4_probe,   /*RJK, this is called from snd_soc_instantiate_card() 
					with *pdev parameter */
        .remove = aic32x4_remove,
        .suspend = aic32x4_suspend,
        .resume = aic32x4_resume,
};

EXPORT_SYMBOL_GPL(soc_codec_dev_aic32x4);

static int __init tlv320aic32x4_modinit(void)
{
	int ret;
	
	dprintk( KERN_INFO "Function : %s \n", __FUNCTION__);

        return snd_soc_register_dai(&tlv320aic32x4_dai);
}

module_init(tlv320aic32x4_modinit);

static void __exit tlv320aic32x4_exit(void)
{
	dprintk( KERN_INFO "Function : %s \n", __FUNCTION__);
        snd_soc_unregister_dai(&tlv320aic32x4_dai);
}

module_exit(tlv320aic32x4_exit);

MODULE_DESCRIPTION("ASoC TLV320AIC32x4 codec driver");
MODULE_AUTHOR("Shahina Shaik <shahina.s@mistralsolutions.com> , Barani Prashanth <gvbarani@mistralsolutions.com");
MODULE_LICENSE("GPL");
